<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access only
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get filter parameters
$batch_year = $_GET['batch_year'] ?? '';
$course = $_GET['course'] ?? '';
$year_level = $_GET['year_level'] ?? '';
$payment_status = $_GET['payment_status'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'name';
$sort_order = $_GET['sort_order'] ?? 'asc';
$show_completed = isset($_GET['show_completed']) && $_GET['show_completed'] === '1';

// Get available academic years for the filter
$academic_years = $conn->query("
    SELECT DISTINCT year_name, id 
    FROM academic_years 
    ORDER BY start_date DESC
");

// Get available courses for the filter
$courses = $conn->query("
    SELECT DISTINCT program_code 
    FROM programs 
    ORDER BY program_code
");

// Build the query based on filters
$query = "
    SELECT 
        s.id,
        s.student_number,
        s.first_name,
        s.last_name,
        p.program_code as course,
        s.year_level,
        u.email,
        u.status,
        ay.year_name as batch_year,
        ay.id as academic_year_id,
        (
            SELECT GROUP_CONCAT(DISTINCT sf_years.academic_year_id)
            FROM student_fees sf_years
            WHERE sf_years.student_id = s.id
        ) as enrolled_years,
        (
            SELECT SUM(sf1.amount)
            FROM student_fees sf1
            WHERE sf1.student_id = s.id
        ) as total_fees,
        (
            SELECT SUM(p1.amount)
            FROM payments p1
            JOIN student_fees sf2 ON p1.student_fee_id = sf2.id
            WHERE sf2.student_id = s.id
            AND p1.status = 'confirmed'
        ) as total_paid,
        (
            SELECT MAX(p2.payment_date)
            FROM payments p2
            JOIN student_fees sf3 ON p2.student_fee_id = sf3.id
            WHERE sf3.student_id = s.id
            AND p2.status = 'confirmed'
        ) as last_payment_date
    FROM students s
    JOIN users u ON s.user_id = u.id
    JOIN programs p ON s.program_id = p.id
    JOIN student_fees sf ON s.id = sf.student_id
    JOIN academic_years ay ON sf.academic_year_id = ay.id
    WHERE 1=1
";

$params = [];
$param_types = "";

if ($batch_year) {
    $query .= " AND ay.year_name = ?";
    $params[] = $batch_year;
    $param_types .= "s";
}

if ($course) {
    $query .= " AND p.program_code = ?";
    $params[] = $course;
    $param_types .= "s";
}

if ($year_level) {
    $query .= " AND s.year_level = ?";
    $params[] = $year_level;
    $param_types .= "i";
}

if ($payment_status) {
    switch ($payment_status) {
        case 'paid':
            $query .= " AND (SELECT SUM(sf1.amount) FROM student_fees sf1 WHERE sf1.student_id = s.id) <= 
                       (SELECT COALESCE(SUM(p1.amount), 0) FROM payments p1 
                        JOIN student_fees sf2 ON p1.student_fee_id = sf2.id 
                        WHERE sf2.student_id = s.id AND p1.status = 'confirmed')";
            break;
        case 'partial':
            $query .= " AND (SELECT COALESCE(SUM(p1.amount), 0) FROM payments p1 
                        JOIN student_fees sf2 ON p1.student_fee_id = sf2.id 
                        WHERE sf2.student_id = s.id AND p1.status = 'confirmed') > 0
                       AND (SELECT SUM(sf1.amount) FROM student_fees sf1 WHERE sf1.student_id = s.id) > 
                       (SELECT COALESCE(SUM(p1.amount), 0) FROM payments p1 
                        JOIN student_fees sf2 ON p1.student_fee_id = sf2.id 
                        WHERE sf2.student_id = s.id AND p1.status = 'confirmed')";
            break;
        case 'unpaid':
            $query .= " AND (SELECT COALESCE(SUM(p1.amount), 0) FROM payments p1 
                        JOIN student_fees sf2 ON p1.student_fee_id = sf2.id 
                        WHERE sf2.student_id = s.id AND p1.status = 'confirmed') = 0";
            break;
    }
}

if (!$show_completed) {
    $query .= " AND s.year_level <= 4";
}

$query .= " GROUP BY s.id";

// Add sorting
switch ($sort_by) {
    case 'name':
        $query .= " ORDER BY s.last_name " . ($sort_order === 'desc' ? 'DESC' : 'ASC');
        break;
    case 'course':
        $query .= " ORDER BY p.program_code " . ($sort_order === 'desc' ? 'DESC' : 'ASC') . ", s.last_name ASC";
        break;
    case 'year_level':
        $query .= " ORDER BY s.year_level " . ($sort_order === 'desc' ? 'DESC' : 'ASC') . ", s.last_name ASC";
        break;
    case 'batch':
        $query .= " ORDER BY ay.year_name " . ($sort_order === 'desc' ? 'DESC' : 'ASC') . ", s.last_name ASC";
        break;
    case 'balance':
        $query .= " ORDER BY (total_fees - COALESCE(total_paid, 0)) " . ($sort_order === 'desc' ? 'DESC' : 'ASC');
        break;
    default:
        $query .= " ORDER BY s.last_name ASC";
}

$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($param_types, ...$params);
}
$stmt->execute();
$students = $stmt->get_result();

$page_title = "Student Batch Report";
require_once '../includes/header.php';

// Calculate summary statistics
$total_students = $students->num_rows;
$total_fees = 0;
$total_paid = 0;
$total_balance = 0;
$payment_status_counts = [
    'paid' => 0,
    'partial' => 0,
    'unpaid' => 0
];

$students_data = [];
while ($row = $students->fetch_assoc()) {
    $row_fees = $row['total_fees'] ?? 0;
    $row_paid = $row['total_paid'] ?? 0;
    $row_balance = $row_fees - $row_paid;
    
    $total_fees += $row_fees;
    $total_paid += $row_paid;
    
    if ($row_balance <= 0) {
        $payment_status_counts['paid']++;
    } elseif ($row_paid > 0) {
        $payment_status_counts['partial']++;
    } else {
        $payment_status_counts['unpaid']++;
    }
    
    $students_data[] = $row;
}
$total_balance = $total_fees - $total_paid;

// Additional accountant-specific summary
$recent_payments_query = "
    SELECT COUNT(*) as count, SUM(amount) as total
    FROM payments
    WHERE status = 'confirmed'
    AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
";
$recent_payments = $conn->query($recent_payments_query)->fetch_assoc();

$pending_payments_query = "
    SELECT COUNT(*) as count, SUM(amount) as total
    FROM payments
    WHERE status = 'pending'
";
$pending_payments = $conn->query($pending_payments_query)->fetch_assoc();
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Financial Report</h6>
                        <div>
                            <button onclick="exportToExcel()" class="btn btn-success btn-sm me-2">
                                <i class="fas fa-file-excel"></i> Export to Excel
                            </button>
                            <button onclick="window.print()" class="btn btn-primary btn-sm">
                                <i class="fas fa-print"></i> Print Report
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Quick Stats for Accountant -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <div class="card bg-gradient-info">
                                <div class="card-body p-3">
                                    <div class="row">
                                        <div class="col-8">
                                            <div class="numbers">
                                                <p class="text-white text-sm mb-0 text-capitalize font-weight-bold">Recent Payments (30 days)</p>
                                                <h5 class="text-white font-weight-bolder mb-0">
                                                    <?php echo formatMoney($recent_payments['total'] ?? 0); ?>
                                                </h5>
                                                <small class="text-white"><?php echo $recent_payments['count']; ?> transactions</small>
                                            </div>
                                        </div>
                                        <div class="col-4 text-end">
                                            <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                <i class="fas fa-money-bill text-info opacity-10"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-warning">
                                <div class="card-body p-3">
                                    <div class="row">
                                        <div class="col-8">
                                            <div class="numbers">
                                                <p class="text-white text-sm mb-0 text-capitalize font-weight-bold">Pending Payments</p>
                                                <h5 class="text-white font-weight-bolder mb-0">
                                                    <?php echo formatMoney($pending_payments['total'] ?? 0); ?>
                                                </h5>
                                                <small class="text-white"><?php echo $pending_payments['count']; ?> transactions</small>
                                            </div>
                                        </div>
                                        <div class="col-4 text-end">
                                            <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                <i class="fas fa-clock text-warning opacity-10"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-success">
                                <div class="card-body p-3">
                                    <div class="row">
                                        <div class="col-8">
                                            <div class="numbers">
                                                <p class="text-white text-sm mb-0 text-capitalize font-weight-bold">Collection Rate</p>
                                                <h5 class="text-white font-weight-bolder mb-0">
                                                    <?php echo $total_fees > 0 ? round(($total_paid / $total_fees) * 100, 1) : 0; ?>%
                                                </h5>
                                                <small class="text-white">Overall collection</small>
                                            </div>
                                        </div>
                                        <div class="col-4 text-end">
                                            <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                <i class="fas fa-chart-pie text-success opacity-10"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-primary">
                                <div class="card-body p-3">
                                    <div class="row">
                                        <div class="col-8">
                                            <div class="numbers">
                                                <p class="text-white text-sm mb-0 text-capitalize font-weight-bold">Outstanding Balance</p>
                                                <h5 class="text-white font-weight-bolder mb-0">
                                                    <?php echo formatMoney($total_balance); ?>
                                                </h5>
                                                <small class="text-white">Total uncollected</small>
                                            </div>
                                        </div>
                                        <div class="col-4 text-end">
                                            <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                <i class="fas fa-wallet text-primary opacity-10"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Filters -->
                    <form method="GET" class="mb-4">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="batch_year" class="form-label">Batch Year</label>
                                    <select class="form-select" id="batch_year" name="batch_year">
                                        <option value="">All Batches</option>
                                        <?php while ($year = $academic_years->fetch_assoc()): ?>
                                            <option value="<?php echo htmlspecialchars($year['year_name']); ?>"
                                                    <?php echo $batch_year === $year['year_name'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($year['year_name']); ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="course" class="form-label">Program</label>
                                    <select class="form-select" id="course" name="course">
                                        <option value="">All Programs</option>
                                        <?php while ($c = $courses->fetch_assoc()): ?>
                                            <option value="<?php echo htmlspecialchars($c['program_code']); ?>"
                                                    <?php echo $course === $c['program_code'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($c['program_code']); ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="year_level" class="form-label">Year Level</label>
                                    <select class="form-select" id="year_level" name="year_level">
                                        <option value="">All Levels</option>
                                        <?php for ($i = 1; $i <= 4; $i++): ?>
                                            <option value="<?php echo $i; ?>" 
                                                    <?php echo $year_level == $i ? 'selected' : ''; ?>>
                                                Year <?php echo $i; ?>
                                            </option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="payment_status" class="form-label">Payment Status</label>
                                    <select class="form-select" id="payment_status" name="payment_status">
                                        <option value="">All Status</option>
                                        <option value="paid" <?php echo $payment_status === 'paid' ? 'selected' : ''; ?>>Fully Paid</option>
                                        <option value="partial" <?php echo $payment_status === 'partial' ? 'selected' : ''; ?>>Partial Payment</option>
                                        <option value="unpaid" <?php echo $payment_status === 'unpaid' ? 'selected' : ''; ?>>Unpaid</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="sort_by" class="form-label">Sort By</label>
                                    <select class="form-select" id="sort_by" name="sort_by">
                                        <option value="name" <?php echo $sort_by === 'name' ? 'selected' : ''; ?>>Name</option>
                                        <option value="course" <?php echo $sort_by === 'course' ? 'selected' : ''; ?>>Program</option>
                                        <option value="year_level" <?php echo $sort_by === 'year_level' ? 'selected' : ''; ?>>Year Level</option>
                                        <option value="batch" <?php echo $sort_by === 'batch' ? 'selected' : ''; ?>>Batch Year</option>
                                        <option value="balance" <?php echo $sort_by === 'balance' ? 'selected' : ''; ?>>Balance</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="sort_order" class="form-label">Sort Order</label>
                                    <select class="form-select" id="sort_order" name="sort_order">
                                        <option value="asc" <?php echo $sort_order === 'asc' ? 'selected' : ''; ?>>Ascending</option>
                                        <option value="desc" <?php echo $sort_order === 'desc' ? 'selected' : ''; ?>>Descending</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label class="form-label d-block">&nbsp;</label>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="show_completed" 
                                               name="show_completed" value="1" <?php echo $show_completed ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="show_completed">
                                            Show Completed Students
                                        </label>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label d-block">&nbsp;</label>
                                <button type="submit" class="btn btn-primary me-2">
                                    <i class="fas fa-filter"></i> Apply Filters
                                </button>
                                <a href="batch_report.php" class="btn btn-secondary">
                                    <i class="fas fa-sync"></i> Reset
                                </a>
                            </div>
                        </div>
                    </form>

                    <!-- Payment Status Summary -->
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <div class="card border-success">
                                <div class="card-body">
                                    <h6 class="card-title text-success">Fully Paid Students</h6>
                                    <h4 class="mb-0"><?php echo $payment_status_counts['paid']; ?></h4>
                                    <small><?php echo $total_students > 0 ? round(($payment_status_counts['paid'] / $total_students) * 100, 1) : 0; ?>% of total students</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card border-warning">
                                <div class="card-body">
                                    <h6 class="card-title text-warning">Partial Payment</h6>
                                    <h4 class="mb-0"><?php echo $payment_status_counts['partial']; ?></h4>
                                    <small><?php echo $total_students > 0 ? round(($payment_status_counts['partial'] / $total_students) * 100, 1) : 0; ?>% of total students</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card border-danger">
                                <div class="card-body">
                                    <h6 class="card-title text-danger">Unpaid Students</h6>
                                    <h4 class="mb-0"><?php echo $payment_status_counts['unpaid']; ?></h4>
                                    <small><?php echo $total_students > 0 ? round(($payment_status_counts['unpaid'] / $total_students) * 100, 1) : 0; ?>% of total students</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Student List -->
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0" id="studentsTable">
                            <thead>
                                <tr>
                                    <th>Student</th>
                                    <th>Program</th>
                                    <th>Year Level</th>
                                    <th>Batch</th>
                                    <th>Total Fees</th>
                                    <th>Total Paid</th>
                                    <th>Balance</th>
                                    <th>Last Payment</th>
                                    <th class="no-export">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($students_data as $student): 
                                    $balance = ($student['total_fees'] ?? 0) - ($student['total_paid'] ?? 0);
                                    $payment_status = '';
                                    if ($balance <= 0) {
                                        $payment_status = 'success';
                                    } elseif ($student['total_paid'] > 0) {
                                        $payment_status = 'warning';
                                    } else {
                                        $payment_status = 'danger';
                                    }
                                ?>
                                <tr>
                                    <td>
                                        <div>
                                            <h6 class="mb-0 text-sm">
                                                <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?>
                                            </h6>
                                            <p class="text-xs text-secondary mb-0">
                                                <?php echo htmlspecialchars($student['student_number']); ?>
                                            </p>
                                        </div>
                                    </td>
                                    <td><?php echo htmlspecialchars($student['course']); ?></td>
                                    <td>Year <?php echo $student['year_level']; ?></td>
                                    <td><?php echo htmlspecialchars($student['batch_year']); ?></td>
                                    <td><?php echo formatMoney($student['total_fees'] ?? 0); ?></td>
                                    <td><?php echo formatMoney($student['total_paid'] ?? 0); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo $payment_status; ?>">
                                            <?php echo formatMoney($balance); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php 
                                        echo $student['last_payment_date'] 
                                            ? date('M j, Y', strtotime($student['last_payment_date']))
                                            : '-';
                                        ?>
                                    </td>
                                    <td class="no-export">
                                        <a href="../payment_history.php?id=<?php echo $student['id']; ?>" 
                                           class="btn btn-info btn-sm" title="View Payment History">
                                            <i class="fas fa-history"></i>
                                        </a>
                                        <a href="review_payment.php?student_id=<?php echo $student['id']; ?>" 
                                           class="btn btn-primary btn-sm" title="Manage Payments">
                                            <i class="fas fa-money-bill"></i>
                                        </a>
                                        <button type="button" 
                                                class="btn btn-warning btn-sm" 
                                                title="Send Payment Reminder"
                                                onclick="sendReminder('<?php echo $student['id']; ?>')">
                                            <i class="fas fa-bell"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print, form {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
    th.no-export, td.no-export {
        display: none;
    }
}
</style>

<script>
function sendReminder(studentId) {
    if (confirm('Are you sure you want to send a payment reminder to this student?')) {
        // TODO: Implement reminder functionality
        alert('Payment reminder sent successfully!');
    }
}

function exportToExcel() {
    let table = document.getElementById("studentsTable");
    let html = table.outerHTML;
    
    // Create a blob with the HTML content
    let blob = new Blob([html], { type: 'application/vnd.ms-excel' });
    let url = window.URL.createObjectURL(blob);
    
    // Create a temporary link and trigger the download
    let a = document.createElement("a");
    a.href = url;
    a.download = 'student_batch_report.xls';
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
}

document.addEventListener('DOMContentLoaded', function() {
    // Auto-submit form when certain filters change
    const autoSubmitFilters = ['batch_year', 'course', 'year_level', 'payment_status', 'sort_by', 'sort_order', 'show_completed'];
    autoSubmitFilters.forEach(filterId => {
        document.getElementById(filterId)?.addEventListener('change', function() {
            this.form.submit();
        });
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>