<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access only
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get date parameters
$date = $_GET['date'] ?? date('Y-m-d');

// Get daily collections
$daily_collections = "
    SELECT 
        p.id,
        p.payment_date,
        p.amount as amount,
        p.payment_method,
        p.reference_number,
        p.status,
        s.student_number,
        s.first_name,
        s.last_name,
        pr.program_code
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN programs pr ON s.program_id = pr.id
    WHERE DATE(p.payment_date) = ?
    AND p.status = 'confirmed'
    ORDER BY p.payment_date DESC
";

$stmt = $conn->prepare($daily_collections);
$stmt->bind_param('s', $date);
$stmt->execute();
$collections = $stmt->get_result();

// Get daily summary
$summary_query = "
    SELECT 
        COUNT(*) as transaction_count,
        SUM(p.amount) as total_amount,
        COUNT(DISTINCT sf.student_id) as unique_students
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE DATE(p.payment_date) = ?
    AND p.status = 'confirmed'
";

$stmt = $conn->prepare($summary_query);
$stmt->bind_param('s', $date);
$stmt->execute();
$summary = $stmt->get_result()->fetch_assoc();

// Get payment methods breakdown
$methods_query = "
    SELECT 
        p.payment_method,
        COUNT(*) as count,
        SUM(p.amount) as total
    FROM payments p
    WHERE DATE(p.payment_date) = ?
    AND p.status = 'confirmed'
    GROUP BY p.payment_method
";

$stmt = $conn->prepare($methods_query);
$stmt->bind_param('s', $date);
$stmt->execute();
$payment_methods = $stmt->get_result();

$page_title = "Daily Collections Report";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Daily Collections Report</h6>
                        <div>
                            <button onclick="exportToExcel()" class="btn btn-success btn-sm me-2">
                                <i class="fas fa-file-excel"></i> Export
                            </button>
                            <button onclick="window.print()" class="btn btn-info btn-sm">
                                <i class="fas fa-print"></i> Print
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Date Selector -->
                    <form method="GET" class="mb-4">
                        <div class="row align-items-center">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="date">Select Date</label>
                                    <input type="date" class="form-control" id="date" name="date" 
                                           value="<?php echo $date; ?>" onchange="this.form.submit()">
                                </div>
                            </div>
                            <div class="col-md-9">
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="card bg-gradient-primary">
                                            <div class="card-body p-3">
                                                <div class="row">
                                                    <div class="col-8">
                                                        <div class="numbers">
                                                            <p class="text-white text-sm mb-0">Total Collections</p>
                                                            <h5 class="text-white font-weight-bolder mb-0">
                                                                <?php echo formatMoney($summary['total_amount'] ?? 0); ?>
                                                            </h5>
                                                        </div>
                                                    </div>
                                                    <div class="col-4 text-end">
                                                        <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                            <i class="fas fa-money-bill text-primary opacity-10"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-gradient-success">
                                            <div class="card-body p-3">
                                                <div class="row">
                                                    <div class="col-8">
                                                        <div class="numbers">
                                                            <p class="text-white text-sm mb-0">Transactions</p>
                                                            <h5 class="text-white font-weight-bolder mb-0">
                                                                <?php echo $summary['transaction_count'] ?? 0; ?>
                                                            </h5>
                                                        </div>
                                                    </div>
                                                    <div class="col-4 text-end">
                                                        <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                            <i class="fas fa-receipt text-success opacity-10"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-gradient-info">
                                            <div class="card-body p-3">
                                                <div class="row">
                                                    <div class="col-8">
                                                        <div class="numbers">
                                                            <p class="text-white text-sm mb-0">Unique Students</p>
                                                            <h5 class="text-white font-weight-bolder mb-0">
                                                                <?php echo $summary['unique_students'] ?? 0; ?>
                                                            </h5>
                                                        </div>
                                                    </div>
                                                    <div class="col-4 text-end">
                                                        <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                            <i class="fas fa-users text-info opacity-10"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>

                    <!-- Payment Methods Breakdown -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <h6 class="mb-3">Payment Methods Breakdown</h6>
                            <div class="table-responsive">
                                <table class="table align-items-center mb-0">
                                    <thead>
                                        <tr>
                                            <th>Payment Method</th>
                                            <th>Transactions</th>
                                            <th>Total Amount</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($method = $payment_methods->fetch_assoc()): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($method['payment_method']); ?></td>
                                            <td><?php echo $method['count']; ?></td>
                                            <td><?php echo formatMoney($method['total']); ?></td>
                                        </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Transactions List -->
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0" id="collectionsTable">
                            <thead>
                                <tr>
                                    <th>Time</th>
                                    <th>Student</th>
                                    <th>Program</th>
                                    <th>Amount</th>
                                    <th>Method</th>
                                    <th>Reference</th>
                                    <th class="no-export">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($payment = $collections->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo date('h:i A', strtotime($payment['payment_date'])); ?></td>
                                    <td>
                                        <div>
                                            <h6 class="mb-0 text-sm">
                                                <?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?>
                                            </h6>
                                            <p class="text-xs text-secondary mb-0">
                                                <?php echo htmlspecialchars($payment['student_number']); ?>
                                            </p>
                                        </div>
                                    </td>
                                    <td><?php echo htmlspecialchars($payment['program_code']); ?></td>
                                    <td><?php echo formatMoney($payment['amount']); ?></td>
                                    <td><?php echo htmlspecialchars($payment['payment_method']); ?></td>
                                    <td><?php echo htmlspecialchars($payment['reference_number']); ?></td>
                                    <td class="no-export">
                                        <a href="review_payment.php?id=<?php echo $payment['id']; ?>" 
                                           class="btn btn-info btn-sm">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <button onclick="printReceipt(<?php echo $payment['id']; ?>)" 
                                                class="btn btn-primary btn-sm">
                                            <i class="fas fa-print"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print, form {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
    th.no-export, td.no-export {
        display: none;
    }
}
</style>

<script>
function exportToExcel() {
    let table = document.getElementById("collectionsTable");
    let html = table.outerHTML;
    
    let blob = new Blob([html], { type: 'application/vnd.ms-excel' });
    let url = window.URL.createObjectURL(blob);
    
    let a = document.createElement("a");
    a.href = url;
    a.download = 'daily_collections_<?php echo $date; ?>.xls';
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
}

function printReceipt(paymentId) {
    // TODO: Implement receipt printing functionality
    alert('Receipt printing functionality will be implemented here.');
}
</script>

<?php require_once '../includes/footer.php'; ?>