<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Pagination setup
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Search functionality
$search_condition = "";
$search_params = [];
$search_types = "";
if (!empty($_GET['search'])) {
    $search = "%" . $_GET['search'] . "%";
    $search_condition = " AND (
        s.student_number LIKE ? OR 
        CONCAT(s.first_name, ' ', s.last_name) LIKE ? OR 
        p.reference_number LIKE ?
    )";
    $search_params = array_fill(0, 3, $search);
    $search_types = "sss";
}

// Get payment statistics
$stats = [
    'total_payments' => 0,
    'pending_payments' => 0,
    'confirmed_payments' => 0,
    'total_amount' => 0,
    'pending_amount' => 0
];

// Get total payments and amounts for today
$stats_query = "
    SELECT 
        COUNT(*) as total_payments,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_count,
        SUM(CASE WHEN status = 'confirmed' THEN 1 ELSE 0 END) as confirmed_count,
        SUM(CASE WHEN status = 'confirmed' THEN amount ELSE 0 END) as total_amount,
        SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END) as pending_amount,
        SUM(CASE WHEN status = 'confirmed' THEN amount ELSE 0 END) as confirmed_amount
    FROM payments
    WHERE DATE(created_at) = CURDATE()";

$result = $conn->query($stats_query);
if ($row = $result->fetch_assoc()) {
    $stats = array_merge($stats, $row);
}

// Get total count for pagination
$count_query = "
    SELECT COUNT(*) as total 
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    WHERE 1=1" . $search_condition;

$stmt = $conn->prepare($count_query);
if (!empty($search_params)) {
    $stmt->bind_param($search_types, ...$search_params);
}
$stmt->execute();
$total_rows = $stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $per_page);

// Get recent payments with search and pagination
$query = "
    SELECT 
        p.*,
        s.student_number,
        s.first_name,
        s.last_name,
        ft.name as fee_type,
        ft.amount as total_fee_amount,
        (
            SELECT SUM(amount) 
            FROM payments 
            WHERE student_fee_id = sf.id 
            AND status = 'confirmed'
        ) as total_paid_amount
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    WHERE 1=1" . $search_condition . "
    ORDER BY p.created_at DESC
    LIMIT ? OFFSET ?";

$stmt = $conn->prepare($query);
if (!empty($search_params)) {
    $params = array_merge($search_params, [$per_page, $offset]);
    $types = $search_types . "ii";
} else {
    $params = [$per_page, $offset];
    $types = "ii";
}
$stmt->bind_param($types, ...$params);
$stmt->execute();
$recent_payments = $stmt->get_result();

// Get payment trends (last 7 days)
$trends = [];
for ($i = 6; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-$i days"));
    $stmt = $conn->prepare("
        SELECT 
            COUNT(*) as count,
            SUM(CASE WHEN status = 'confirmed' THEN amount ELSE 0 END) as amount
        FROM payments
        WHERE DATE(created_at) = ?
    ");
    $stmt->bind_param("s", $date);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $trends[$date] = [
        'count' => $row['count'],
        'amount' => $row['amount'] ?? 0
    ];
}

$page_title = "Accountant Dashboard";
require_once '../includes/header.php';
?>

<style>
.payment-row {
    transition: background-color 0.3s ease;
}
.payment-row.highlight {
    background-color: #fff3cd;
}
.pagination .page-link {
    padding: 0.5rem 0.75rem;
    margin-left: -1px;
    color: #007bff;
    background-color: #fff;
    border: 1px solid #dee2e6;
}
.pagination .page-item.active .page-link {
    color: #fff;
    background-color: #007bff;
    border-color: #007bff;
}
.pagination .page-item.disabled .page-link {
    color: #6c757d;
    pointer-events: none;
    background-color: #fff;
    border-color: #dee2e6;
}
.status-badge {
    padding: 0.35em 0.65em;
    font-size: 0.75em;
    font-weight: 700;
    border-radius: 0.25rem;
}
.status-badge.confirmed { background-color: #28a745; color: white; }
.status-badge.pending { background-color: #ffc107; }
.status-badge.failed { background-color: #dc3545; color: white; }
</style>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Accountant Dashboard</h2>
        <a href="../index.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back
        </a>
    </div>

    <!-- Statistics Cards -->
    <div class="row g-3 mb-4">
        <div class="col-md-4">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">Today's Payments</h6>
                    <h2 class="card-title mb-0">GHs <?php echo number_format($stats['total_amount'], 2); ?></h2>
                    <small class="text-success">
                        <?php echo $stats['confirmed_count']; ?> confirmed payments
                    </small>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">Pending Payments</h6>
                    <h2 class="card-title mb-0">GHs <?php echo number_format($stats['pending_amount'], 2); ?></h2>
                    <small class="text-warning">
                        <?php echo $stats['pending_count']; ?> pending payments
                    </small>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Recent Payments -->
        <div class="col-md-8 mb-4">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0">Recent Payments</h5>
                        <a href="dashboard.php" class="btn btn-primary btn-sm">Refresh</a>
                    </div>
                    <!-- Search form -->
                    <form method="get" class="mb-3">
                        <div class="input-group">
                            <input type="text" name="search" class="form-control" 
                                value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search']) : ''; ?>"
                                placeholder="Search by student number, name, or reference...">
                            <button type="submit" class="btn btn-outline-primary">Search</button>
                            <?php if (!empty($_GET['search'])): ?>
                                <a href="?" class="btn btn-outline-secondary">Clear</a>
                            <?php endif; ?>
                        </div>
                    </form>
                    <?php if ($total_rows > 0): ?>
                        <div class="text-muted small">
                            Showing <?php echo ($offset + 1); ?> to <?php echo min($offset + $per_page, $total_rows); ?> 
                            of <?php echo $total_rows; ?> payments
                        </div>
                    <?php endif; ?>
                </div>
                <div class="table-responsive">
                    <table class="table align-middle mb-0">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Student</th>
                                <th class="text-end">Amount</th>
                                <th>Status</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($recent_payments->num_rows === 0): ?>
                                <tr>
                                    <td colspan="5" class="text-center py-3">
                                        <?php echo !empty($_GET['search']) ? 'No payments found matching your search.' : 'No recent payments found.'; ?>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php while ($payment = $recent_payments->fetch_assoc()): ?>
                                    <tr class="payment-row">
                                        <td>
                                            <div><?php echo date('M d, Y', strtotime($payment['created_at'])); ?></div>
                                            <small class="text-muted"><?php echo date('g:i A', strtotime($payment['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?></div>
                                            <small class="text-muted"><?php echo $payment['student_number']; ?></small>
                                        </td>
                                        <td class="text-end">GHs <?php echo number_format($payment['amount'], 2); ?></td>
                                        <td>
                                            <span class="status-badge <?php echo $payment['status']; ?>">
                                                <?php echo ucfirst($payment['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($payment['status'] === 'pending'): ?>
                                                <a href="review_payment.php?id=<?php echo $payment['id']; ?>" 
                                                   class="btn btn-primary btn-sm">
                                                    Review
                                                </a>
                                            <?php else: ?>
                                                <a href="review_payment.php?id=<?php echo $payment['id']; ?>" 
                                                   class="btn btn-outline-secondary btn-sm">
                                                    Details
                                                </a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($total_pages > 1): ?>
                <div class="card-footer">
                    <nav>
                        <ul class="pagination justify-content-center mb-0">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page-1; ?><?php 
                                        echo !empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : ''; 
                                    ?>">&laquo;</a>
                                </li>
                            <?php endif; ?>
                            
                            <?php
                            // Show up to 5 page numbers, centered around current page
                            $start_page = max(1, $page - 2);
                            $end_page = min($total_pages, $page + 2);
                            
                            if ($start_page > 1) {
                                echo '<li class="page-item"><a class="page-link" href="?page=1'.
                                    (!empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : '').'">1</a></li>';
                                if ($start_page > 2) {
                                    echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                }
                            }
                            
                            for ($i = $start_page; $i <= $end_page; $i++) {
                                echo '<li class="page-item '.($i == $page ? 'active' : '').'">
                                    <a class="page-link" href="?page='.$i.
                                    (!empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : '').
                                    '">'.$i.'</a></li>';
                            }
                            
                            if ($end_page < $total_pages) {
                                if ($end_page < $total_pages - 1) {
                                    echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                }
                                echo '<li class="page-item"><a class="page-link" href="?page='.$total_pages.
                                    (!empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : '').
                                    '">'.$total_pages.'</a></li>';
                            }
                            ?>
                            
                            <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page+1; ?><?php 
                                        echo !empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : ''; 
                                    ?>">&raquo;</a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Payment Trends -->
        <div class="col-md-4 mb-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Payment Trends (Last 7 Days)</h5>
                </div>
                <div class="card-body">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th class="text-end">Count</th>
                                <th class="text-end">Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($trends as $date => $data): ?>
                                <tr>
                                    <td><?php echo date('M d', strtotime($date)); ?></td>
                                    <td class="text-end"><?php echo $data['count']; ?></td>
                                    <td class="text-end">GHs <?php echo number_format($data['amount'], 2); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Highlight new payments
document.addEventListener('DOMContentLoaded', function() {
    const urlParams = new URLSearchParams(window.location.search);
    const highlight = urlParams.get('highlight');
    if (highlight) {
        const row = document.querySelector(`tr[data-payment-id="${highlight}"]`);
        if (row) {
            row.classList.add('highlight');
            row.scrollIntoView({ behavior: 'smooth', block: 'center' });
            setTimeout(() => row.classList.remove('highlight'), 3000);
        }
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>