<?php
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/payment_logger.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    http_response_code(403);
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized access']);
    exit();
}

header('Content-Type: application/json');

if (!isset($_GET['reference'])) {
    echo json_encode(['status' => 'error', 'message' => 'Reference number is required']);
    exit();
}

$reference = $_GET['reference'];

// Get payment details
$stmt = $conn->prepare("
    SELECT 
        p.*,
        sf.student_id,
        s.first_name,
        s.last_name,
        s.student_number,
        ft.name as fee_type,
        ft.amount as fee_amount,
        ay.year_name,
        pl.response_data,
        pl.created_at as log_date
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    JOIN academic_years ay ON sf.academic_year_id = ay.id
    LEFT JOIN payment_logs pl ON p.id = pl.payment_id
    WHERE p.reference_number = ?
    ORDER BY pl.created_at DESC
    LIMIT 1
");

$stmt->bind_param("s", $reference);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();

if (!$payment) {
    echo json_encode(['status' => 'error', 'message' => 'Payment not found']);
    exit();
}

// Build HTML content
$html = "
<div class='container-fluid'>
    <div class='row'>
        <div class='col-md-6'>
            <h6>Student Information</h6>
            <table class='table table-sm'>
                <tr>
                    <th>Student Number</th>
                    <td>{$payment['student_number']}</td>
                </tr>
                <tr>
                    <th>Name</th>
                    <td>{$payment['first_name']} {$payment['last_name']}</td>
                </tr>
            </table>
        </div>
        <div class='col-md-6'>
            <h6>Payment Information</h6>
            <table class='table table-sm'>
                <tr>
                    <th>Reference</th>
                    <td>{$payment['reference_number']}</td>
                </tr>
                <tr>
                    <th>Amount</th>
                    <td>" . formatMoney($payment['amount']) . "</td>
                </tr>
                <tr>
                    <th>Status</th>
                    <td><span class='badge bg-" . getStatusBadgeClass($payment['status']) . "'>" . 
                        ucfirst($payment['status']) . "</span></td>
                </tr>
                <tr>
                    <th>Payment Date</th>
                    <td>" . date('Y-m-d H:i:s', strtotime($payment['payment_date'])) . "</td>
                </tr>
            </table>
        </div>
    </div>
    <div class='row mt-3'>
        <div class='col-12'>
            <h6>Fee Information</h6>
            <table class='table table-sm'>
                <tr>
                    <th>Fee Type</th>
                    <td>{$payment['fee_type']}</td>
                </tr>
                <tr>
                    <th>Academic Year</th>
                    <td>{$payment['year_name']}</td>
                </tr>
                <tr>
                    <th>Total Fee Amount</th>
                    <td>" . formatMoney($payment['fee_amount']) . "</td>
                </tr>
            </table>
        </div>
    </div>
";

// Add payment gateway response if available
if ($payment['response_data']) {
    $response = json_decode($payment['response_data'], true);
    $html .= "
    <div class='row mt-3'>
        <div class='col-12'>
            <h6>Payment Gateway Response</h6>
            <div class='alert alert-info'>
                <pre class='mb-0'>" . json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . "</pre>
            </div>
        </div>
    </div>
    ";
}

$html .= "</div>";

echo json_encode([
    'status' => 'success',
    'html' => $html
]);

function getStatusBadgeClass($status) {
    switch(strtolower($status)) {
        case 'success':
        case 'confirmed':
            return 'success';
        case 'pending':
            return 'warning';
        case 'failed':
        case 'rejected':
            return 'danger';
        default:
            return 'secondary';
    }
}
?>