<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

$success_message = '';
$error_message = '';

// Handle academic year operations (Create, Update, Delete)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $year_name = trim($_POST['year_name'] ?? '');
            $start_date = trim($_POST['start_date'] ?? '');
            $end_date = trim($_POST['end_date'] ?? '');
            $status = $_POST['status'] ?? 'active';
            
            if (empty($year_name) || empty($start_date) || empty($end_date)) {
                $error_message = "All fields are required.";
            } else {
                // Validate dates
                $start = new DateTime($start_date);
                $end = new DateTime($end_date);
                
                if ($end <= $start) {
                    $error_message = "End date must be after start date.";
                } else {
                    $stmt = $conn->prepare("
                        INSERT INTO academic_years (
                            year_name, 
                            start_date, 
                            end_date,
                            status
                        ) VALUES (?, ?, ?, ?)
                    ");
                    $stmt->bind_param("ssss", $year_name, $start_date, $end_date, $status);
                    
                    if ($stmt->execute()) {
                        $success_message = "Academic year created successfully!";
                    } else {
                        $error_message = "Error creating academic year: " . $conn->error;
                    }
                }
            }
            break;
            
        case 'update':
            $year_id = intval($_POST['year_id'] ?? 0);
            $year_name = trim($_POST['year_name'] ?? '');
            $start_date = trim($_POST['start_date'] ?? '');
            $end_date = trim($_POST['end_date'] ?? '');
            $status = $_POST['status'] ?? 'active';
            
            if ($year_id <= 0 || empty($year_name) || empty($start_date) || empty($end_date)) {
                $error_message = "Invalid academic year data.";
            } else {
                // Validate dates
                $start = new DateTime($start_date);
                $end = new DateTime($end_date);
                
                if ($end <= $start) {
                    $error_message = "End date must be after start date.";
                } else {
                    $stmt = $conn->prepare("
                        UPDATE academic_years 
                        SET year_name = ?,
                            start_date = ?,
                            end_date = ?,
                            status = ?
                        WHERE id = ?
                    ");
                    $stmt->bind_param("ssssi", $year_name, $start_date, $end_date, $status, $year_id);
                    
                    if ($stmt->execute()) {
                        $success_message = "Academic year updated successfully!";
                    } else {
                        $error_message = "Error updating academic year: " . $conn->error;
                    }
                }
            }
            break;
            
        case 'delete':
            $year_id = intval($_POST['year_id'] ?? 0);
            
            // Check if academic year is being used in programs
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM programs WHERE academic_year_id = ?");
            $stmt->bind_param("i", $year_id);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            
            if ($result['count'] > 0) {
                $error_message = "Cannot delete academic year: It is used in " . $result['count'] . " program(s).";
            } else {
                $stmt = $conn->prepare("DELETE FROM academic_years WHERE id = ?");
                $stmt->bind_param("i", $year_id);
                
                if ($stmt->execute()) {
                    $success_message = "Academic year deleted successfully!";
                } else {
                    $error_message = "Error deleting academic year: " . $conn->error;
                }
            }
            break;
    }
}

// Get all academic years with program counts
$academic_years = $conn->query("
    SELECT 
        ay.*,
        (SELECT COUNT(*) FROM programs WHERE academic_year_id = ay.id) as program_count
    FROM academic_years ay
    ORDER BY start_date DESC
");

if (!$academic_years) {
    // Create academic_years table if it doesn't exist
    $create_table = $conn->query("
        CREATE TABLE IF NOT EXISTS academic_years (
            id INT AUTO_INCREMENT PRIMARY KEY,
            year_name VARCHAR(50) NOT NULL UNIQUE,
            start_date DATE NOT NULL,
            end_date DATE NOT NULL,
            status ENUM('active', 'inactive') DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    if ($create_table) {
        $academic_years = $conn->query("
            SELECT 
                ay.*,
                (SELECT COUNT(*) FROM programs WHERE academic_year_id = ay.id) as program_count
            FROM academic_years ay
            ORDER BY start_date DESC
        ");
    }
}

$page_title = "Manage Academic Years";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Manage Academic Years</h5>
                    <button type="button" class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addAcademicYearModal">
                        <i class="fas fa-plus"></i> Add New Academic Year
                    </button>
                </div>
                <div class="card-body">
                    <?php if ($success_message): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?php echo $success_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if ($error_message): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?php echo $error_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Academic Year</th>
                                    <th>Start Date</th>
                                    <th>End Date</th>
                                    <th>Programs</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($academic_years && $academic_years->num_rows > 0): ?>
                                    <?php while ($year = $academic_years->fetch_assoc()): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($year['year_name']); ?></td>
                                            <td><?php echo date('M d, Y', strtotime($year['start_date'])); ?></td>
                                            <td><?php echo date('M d, Y', strtotime($year['end_date'])); ?></td>
                                            <td><?php echo $year['program_count']; ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $year['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($year['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-info" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editAcademicYearModal<?php echo $year['id']; ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <?php if ($year['program_count'] == 0): ?>
                                                    <button type="button" class="btn btn-sm btn-danger" 
                                                            onclick="deleteAcademicYear(<?php echo $year['id']; ?>)">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </td>
                                        </tr>

                                        <!-- Edit Academic Year Modal -->
                                        <div class="modal fade" id="editAcademicYearModal<?php echo $year['id']; ?>" tabindex="-1">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <form method="POST" action="">
                                                        <input type="hidden" name="action" value="update">
                                                        <input type="hidden" name="year_id" value="<?php echo $year['id']; ?>">
                                                        
                                                        <div class="modal-header">
                                                            <h5 class="modal-title">Edit Academic Year</h5>
                                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                        </div>
                                                        
                                                        <div class="modal-body">
                                                            <div class="mb-3">
                                                                <label class="form-label">Academic Year Name</label>
                                                                <input type="text" name="year_name" class="form-control" 
                                                                       value="<?php echo htmlspecialchars($year['year_name']); ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Start Date</label>
                                                                <input type="date" name="start_date" class="form-control" 
                                                                       value="<?php echo $year['start_date']; ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">End Date</label>
                                                                <input type="date" name="end_date" class="form-control" 
                                                                       value="<?php echo $year['end_date']; ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Status</label>
                                                                <select name="status" class="form-select" required>
                                                                    <option value="active" <?php echo $year['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                                                    <option value="inactive" <?php echo $year['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                                                </select>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                            <button type="submit" class="btn btn-primary">Update Academic Year</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="text-center">
                                            <p class="my-3">No academic years found. Click "Add New Academic Year" to create one.</p>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Academic Year Modal -->
<div class="modal fade" id="addAcademicYearModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="action" value="create">
                
                <div class="modal-header">
                    <h5 class="modal-title">Add New Academic Year</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Academic Year Name</label>
                        <input type="text" name="year_name" class="form-control" 
                               placeholder="e.g., 2025-2026" required>
                        <small class="text-muted">Example: 2025-2026</small>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Start Date</label>
                        <input type="date" name="start_date" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">End Date</label>
                        <input type="date" name="end_date" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select" required>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Create Academic Year</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Academic Year Form (Hidden) -->
<form id="deleteAcademicYearForm" method="POST" style="display: none;">
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="year_id" id="deleteYearId">
</form>

<script>
function deleteAcademicYear(yearId) {
    if (confirm('Are you sure you want to delete this academic year? This action cannot be undone.')) {
        document.getElementById('deleteYearId').value = yearId;
        document.getElementById('deleteAcademicYearForm').submit();
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>