<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';
require_once '../includes/database.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get database connection
$db = Database::getInstance();
$conn = $db->getConnection();

// Initialize messages
$success_message = '';
$error_message = '';

if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $name = trim($_POST['name'] ?? '');
            $level = trim($_POST['level'] ?? '');
            $amount = floatval($_POST['amount'] ?? 0);
            $due_date = trim($_POST['due_date'] ?? '');
            $status = 'active'; // Set default status
            
            if (empty($name) || $amount <= 0 || empty($due_date) || empty($level)) {
                $_SESSION['error_message'] = "Name, level, amount, and due date are required. Amount must be greater than 0.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            // Validate level (must be Level 100-400)
            $valid_levels = ['Level 100', 'Level 200', 'Level 300', 'Level 400'];
            if (!in_array($level, $valid_levels)) {
                $_SESSION['error_message'] = "Invalid level. Level must be between Level 100 and Level 400.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }            // Check if fee type name already exists
            $check_stmt = $conn->prepare("SELECT id FROM fee_types WHERE name = ?");
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("s", $name);
            $check_stmt->execute();
            $result = $check_stmt->get_result();
            
            if ($result->num_rows > 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "A fee type with this name already exists.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();
            
            // Create new fee type
            $stmt = $conn->prepare("
                INSERT INTO fee_types (name, level, amount, due_date, status) 
                VALUES (?, ?, ?, ?, 'active')
            ");
            
            if ($stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $stmt->bind_param("ssds", $name, $level, $amount, $due_date);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Fee type created successfully!";
            } else {
                $_SESSION['error_message'] = "Error creating fee type: " . $stmt->error;
            }
            $stmt->close();
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
            break;
            
        case 'update':
            $fee_id = intval($_POST['fee_id'] ?? 0);
            $name = trim($_POST['name'] ?? '');
            $level = trim($_POST['level'] ?? '');
            $amount = floatval($_POST['amount'] ?? 0);
            $status = $_POST['status'] ?? 'active';
            
            if ($fee_id <= 0 || empty($name) || $amount <= 0) {
                $_SESSION['error_message'] = "Invalid fee type data.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            // Check if updated name conflicts with other fee types
            $check_stmt = $conn->prepare("SELECT id FROM fee_types WHERE name = ? AND id != ?");
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("si", $name, $fee_id);
            $check_stmt->execute();
            $result = $check_stmt->get_result();
            
            if ($result->num_rows > 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "Another fee type with this name already exists.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();
            
            // Update fee type
            $stmt = $conn->prepare("
                UPDATE fee_types 
                SET name = ?, level = ?, amount = ?, status = ?
                WHERE id = ?
            ");
            
            if ($stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $stmt->bind_param("ssdsi", $name, $level, $amount, $status, $fee_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Fee type updated successfully!";
            } else {
                $_SESSION['error_message'] = "Error updating fee type: " . $stmt->error;
            }
            $stmt->close();
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
            break;
            
        case 'delete':
            $fee_id = intval($_POST['fee_id'] ?? 0);
            
            if ($fee_id <= 0) {
                $_SESSION['error_message'] = "Invalid fee type ID.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            // Check if fee type is in use
            $check_stmt = $conn->prepare("
                SELECT COUNT(*) as count 
                FROM student_fees 
                WHERE fee_type_id = ?
            ");
            
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("i", $fee_id);
            $check_stmt->execute();
            $result = $check_stmt->get_result()->fetch_assoc();
            
            if ($result['count'] > 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "Cannot delete: This fee type is being used in student bills.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();
            
            // Delete fee type
            $stmt = $conn->prepare("DELETE FROM fee_types WHERE id = ?");
            
            if ($stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $stmt->bind_param("i", $fee_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Fee type deleted successfully!";
            } else {
                $_SESSION['error_message'] = "Error deleting fee type: " . $stmt->error;
            }
            $stmt->close();
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
            break;
    }
}

// Create fee_types table if it doesn't exist
$create_table = $conn->query("
    CREATE TABLE IF NOT EXISTS fee_types (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        level VARCHAR(50),
        amount DECIMAL(10,2) NOT NULL DEFAULT 0.00,
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
");

// Pagination setup
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Get total number of fee types for pagination
$total_query = $conn->query("SELECT COUNT(*) as total FROM fee_types");
$total_rows = $total_query->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $per_page);

// Get fee types for current page
$fee_types = $conn->query("
    SELECT * FROM fee_types 
    ORDER BY name
    LIMIT $per_page OFFSET $offset
");

$page_title = "Manage Fee Types";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Manage Fee Types</h5>
                    <button type="button" class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addFeeTypeModal">
                        <i class="fas fa-plus"></i> Add New Fee Type
                    </button>
                </div>
                <div class="card-body">
                    <?php if ($success_message): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?php echo $success_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if ($error_message): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?php echo $error_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Level</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($fee_types && $fee_types->num_rows > 0): ?>
                                    <?php while ($fee = $fee_types->fetch_assoc()): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($fee['name']); ?></td>
                                            <td>
                                                <small class="text-muted">
                                                    <?php echo !empty($fee['level']) ? htmlspecialchars($fee['level']) : 'No level assigned'; ?>
                                                </small>
                                            </td>
                                            <td>GHs <?php echo number_format($fee['amount'], 2); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $fee['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($fee['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-info" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editFeeTypeModal<?php echo $fee['id']; ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-danger" 
                                                        onclick="deleteFeeType(<?php echo $fee['id']; ?>)">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </td>
                                        </tr>

                                        <!-- Edit Fee Type Modal -->
                                        <div class="modal fade" id="editFeeTypeModal<?php echo $fee['id']; ?>" tabindex="-1">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <form method="POST" action="">
                                                        <input type="hidden" name="action" value="update">
                                                        <input type="hidden" name="fee_id" value="<?php echo $fee['id']; ?>">
                                                        
                                                        <div class="modal-header">
                                                            <h5 class="modal-title">Edit Fee Type</h5>
                                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                        </div>
                                                        
                                                        <div class="modal-body">
                                                            <div class="mb-3">
                                                                <label class="form-label">Name</label>
                                                                <input type="text" name="name" class="form-control" 
                                                                       value="<?php echo htmlspecialchars($fee['name']); ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Level</label>
                                                                <select name="level" class="form-select" required>
                                                                    <option value="">Select Level</option>
                                                                    <option value="Level 100" <?php echo $fee['level'] === 'Level 100' ? 'selected' : ''; ?>>Level 100</option>
                                                                    <option value="Level 200" <?php echo $fee['level'] === 'Level 200' ? 'selected' : ''; ?>>Level 200</option>
                                                                    <option value="Level 300" <?php echo $fee['level'] === 'Level 300' ? 'selected' : ''; ?>>Level 300</option>
                                                                    <option value="Level 400" <?php echo $fee['level'] === 'Level 400' ? 'selected' : ''; ?>>Level 400</option>
                                                                </select>
                                                            </div>

                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Amount (GHs)</label>
                                                                <input type="number" name="amount" class="form-control" 
                                                                       value="<?php echo $fee['amount']; ?>" step="0.01" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Status</label>
                                                                <select name="status" class="form-select" required>
                                                                    <option value="active" <?php echo $fee['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                                                    <option value="inactive" <?php echo $fee['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                                                </select>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                            <button type="submit" class="btn btn-primary">Update Fee Type</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center">
                                            <p class="my-3">No fee types found. Click "Add New Fee Type" to create one.</p>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <?php if ($total_pages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <?php if ($page > 1): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?php echo ($page - 1); ?>">Previous</a>
                                        </li>
                                    <?php endif; ?>
                                    
                                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    
                                    <?php if ($page < $total_pages): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?php echo ($page + 1); ?>">Next</a>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Fee Type Modal -->
<div class="modal fade" id="addFeeTypeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="action" value="create">
                
                <div class="modal-header">
                    <h5 class="modal-title">Add New Fee Type</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Name</label>
                        <input type="text" name="name" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Level</label>
                        <select name="level" class="form-select" required>
                            <option value="">Select Level</option>
                            <option value="Level 100">Level 100</option>
                            <option value="Level 200">Level 200</option>
                            <option value="Level 300">Level 300</option>
                            <option value="Level 400">Level 400</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Level</label>
                        <input type="number" name="level" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Amount (GHs)</label>
                        <input type="number" name="amount" class="form-control" step="0.01" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Due Date</label>
                        <input type="date" name="due_date" class="form-control" required>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Create Fee Type</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Fee Type Form (Hidden) -->
<form id="deleteFeeTypeForm" method="POST" style="display: none;">
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="fee_id" id="deleteFeeTypeId">
</form>

<script>
function deleteFeeType(feeId) {
    if (confirm('Are you sure you want to delete this fee type? This action cannot be undone.')) {
        document.getElementById('deleteFeeTypeId').value = feeId;
        document.getElementById('deleteFeeTypeForm').submit();
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>