<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Handle refund request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'initiate_refund') {
        $payment_id = $_POST['payment_id'];
        $refund_amount = $_POST['refund_amount'];
        $refund_reason = $_POST['refund_reason'];

        // Validate refund amount
        $stmt = $conn->prepare("SELECT amount, status FROM payments WHERE id = ?");
        $stmt->bind_param("i", $payment_id);
        $stmt->execute();
        $payment = $stmt->get_result()->fetch_assoc();

        if ($payment && $payment['status'] === 'confirmed' && $refund_amount <= $payment['amount']) {
            // Update payment status and refund amount
            $stmt = $conn->prepare("
                UPDATE payments 
                SET status = 'refunded',
                    refunded_amount = ?,
                    refund_reason = ?,
                    refund_date = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->bind_param("dsi", $refund_amount, $refund_reason, $payment_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Refund processed successfully.";
            } else {
                $_SESSION['error_message'] = "Error processing refund.";
            }
        } else {
            $_SESSION['error_message'] = "Invalid refund request.";
        }
        
        header('Location: manage_refunds.php');
        exit();
    }
}

// Pagination setup
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Get payments that can be refunded
$query = "
    SELECT 
        p.*,
        s.student_number,
        s.first_name,
        s.last_name,
        ft.name as fee_type,
        ft.amount as fee_amount
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    WHERE p.status IN ('confirmed', 'refunded')
    ORDER BY p.payment_date DESC
    LIMIT ? OFFSET ?";

$stmt = $conn->prepare($query);
$stmt->bind_param("ii", $per_page, $offset);
$stmt->execute();
$payments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get total count for pagination
$count_query = "
    SELECT COUNT(*) as total 
    FROM payments p
    WHERE p.status IN ('confirmed', 'refunded')";

$total_rows = $conn->query($count_query)->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $per_page);

$page_title = "Manage Refunds";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php 
            echo $_SESSION['success_message'];
            unset($_SESSION['success_message']);
            ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?php 
            echo $_SESSION['error_message'];
            unset($_SESSION['error_message']);
            ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0">Payment Refunds</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Student</th>
                            <th>Payment Date</th>
                            <th>Reference</th>
                            <th>Fee Type</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Refunded Amount</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($payments as $payment): ?>
                            <tr>
                                <td>
                                    <?php echo htmlspecialchars($payment['student_number']); ?><br>
                                    <small><?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?></small>
                                </td>
                                <td><?php echo date('Y-m-d H:i', strtotime($payment['payment_date'])); ?></td>
                                <td><?php echo htmlspecialchars($payment['reference_number']); ?></td>
                                <td><?php echo htmlspecialchars($payment['fee_type']); ?></td>
                                <td><?php echo formatMoney($payment['amount']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $payment['status'] === 'confirmed' ? 'success' : 'warning'; ?>">
                                        <?php echo ucfirst($payment['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($payment['status'] === 'refunded'): ?>
                                        <?php echo formatMoney($payment['refunded_amount']); ?>
                                        <br><small class="text-muted">
                                            <?php echo htmlspecialchars($payment['refund_reason']); ?>
                                        </small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($payment['status'] === 'confirmed'): ?>
                                        <button type="button" 
                                                class="btn btn-warning btn-sm"
                                                data-bs-toggle="modal"
                                                data-bs-target="#refundModal"
                                                data-payment-id="<?php echo $payment['id']; ?>"
                                                data-max-amount="<?php echo $payment['amount']; ?>"
                                                data-reference="<?php echo $payment['reference_number']; ?>">
                                            Process Refund
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <nav aria-label="Page navigation" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Refund Modal -->
<div class="modal fade" id="refundModal" tabindex="-1" aria-labelledby="refundModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="refundModalLabel">Process Refund</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="initiate_refund">
                    <input type="hidden" name="payment_id" id="payment_id">
                    
                    <div class="mb-3">
                        <label for="reference_number" class="form-label">Payment Reference</label>
                        <input type="text" class="form-control" id="reference_number" readonly>
                    </div>

                    <div class="mb-3">
                        <label for="refund_amount" class="form-label">Refund Amount</label>
                        <input type="number" step="0.01" class="form-control" id="refund_amount" name="refund_amount" required>
                        <small class="form-text text-muted">Maximum amount: <span id="max_amount"></span></small>
                    </div>

                    <div class="mb-3">
                        <label for="refund_reason" class="form-label">Reason for Refund</label>
                        <textarea class="form-control" id="refund_reason" name="refund_reason" required></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-warning">Process Refund</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var refundModal = document.getElementById('refundModal');
    refundModal.addEventListener('show.bs.modal', function(event) {
        var button = event.relatedTarget;
        var paymentId = button.getAttribute('data-payment-id');
        var maxAmount = button.getAttribute('data-max-amount');
        var reference = button.getAttribute('data-reference');
        
        var modal = this;
        modal.querySelector('#payment_id').value = paymentId;
        modal.querySelector('#reference_number').value = reference;
        modal.querySelector('#max_amount').textContent = maxAmount;
        modal.querySelector('#refund_amount').max = maxAmount;
        modal.querySelector('#refund_amount').value = maxAmount;
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>