<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access only
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get month and year parameters
$month = $_GET['month'] ?? date('m');
$year = $_GET['year'] ?? date('Y');
$selected_date = sprintf('%s-%s-01', $year, $month);

// Get monthly collections
$monthly_collections = "
    SELECT 
        DATE(p.payment_date) as payment_date,
        COUNT(*) as transactions,
        SUM(p.amount) as daily_total,
        COUNT(DISTINCT sf.student_id) as unique_students
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE MONTH(p.payment_date) = ?
    AND YEAR(p.payment_date) = ?
    AND p.status = 'confirmed'
    GROUP BY DATE(p.payment_date)
    ORDER BY payment_date
";

$stmt = $conn->prepare($monthly_collections);
$stmt->bind_param('ss', $month, $year);
$stmt->execute();
$collections = $stmt->get_result();

// Get monthly summary
$summary_query = "
    SELECT 
        COUNT(*) as total_transactions,
        SUM(p.amount) as total_amount,
        COUNT(DISTINCT sf.student_id) as total_students,
        AVG(p.amount) as average_payment
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE MONTH(p.payment_date) = ?
    AND YEAR(p.payment_date) = ?
    AND p.status = 'confirmed'
";

$stmt = $conn->prepare($summary_query);
$stmt->bind_param('ss', $month, $year);
$stmt->execute();
$summary = $stmt->get_result()->fetch_assoc();

// Get payment methods breakdown
$methods_query = "
    SELECT 
        payment_method,
        COUNT(*) as count,
        SUM(amount) as total
    FROM payments
    WHERE MONTH(payment_date) = ?
    AND YEAR(payment_date) = ?
    AND status = 'confirmed'
    GROUP BY payment_method
";

$stmt = $conn->prepare($methods_query);
$stmt->bind_param('ss', $month, $year);
$stmt->execute();
$payment_methods = $stmt->get_result();

// Get program breakdown
$program_query = "
    SELECT 
        p.program_code,
        COUNT(DISTINCT sf.student_id) as student_count,
        SUM(py.amount) as total_amount
    FROM payments py
    JOIN student_fees sf ON py.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN programs p ON s.program_id = p.id
    WHERE MONTH(py.payment_date) = ?
    AND YEAR(py.payment_date) = ?
    AND py.status = 'confirmed'
    GROUP BY p.program_code
    ORDER BY total_amount DESC
";

$stmt = $conn->prepare($program_query);
$stmt->bind_param('ss', $month, $year);
$stmt->execute();
$programs = $stmt->get_result();

$page_title = "Monthly Summary Report";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Monthly Summary Report</h6>
                        <div>
                            <button onclick="exportToExcel()" class="btn btn-success btn-sm me-2">
                                <i class="fas fa-file-excel"></i> Export
                            </button>
                            <button onclick="window.print()" class="btn btn-info btn-sm">
                                <i class="fas fa-print"></i> Print
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Month Selector -->
                    <form method="GET" class="mb-4">
                        <div class="row align-items-center">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="month_year">Select Month and Year</label>
                                    <input type="month" class="form-control" id="month_year" name="month_year" 
                                           value="<?php echo sprintf('%s-%s', $year, $month); ?>" 
                                           onchange="updateMonthYear(this.value)">
                                </div>
                            </div>
                            <div class="col-md-8">
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="card bg-gradient-primary">
                                            <div class="card-body p-3">
                                                <div class="numbers">
                                                    <p class="text-white text-sm mb-0">Total Collections</p>
                                                    <h5 class="text-white font-weight-bolder mb-0">
                                                        <?php echo formatMoney($summary['total_amount'] ?? 0); ?>
                                                    </h5>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-gradient-success">
                                            <div class="card-body p-3">
                                                <div class="numbers">
                                                    <p class="text-white text-sm mb-0">Transactions</p>
                                                    <h5 class="text-white font-weight-bolder mb-0">
                                                        <?php echo $summary['total_transactions'] ?? 0; ?>
                                                    </h5>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-gradient-info">
                                            <div class="card-body p-3">
                                                <div class="numbers">
                                                    <p class="text-white text-sm mb-0">Unique Students</p>
                                                    <h5 class="text-white font-weight-bolder mb-0">
                                                        <?php echo $summary['total_students'] ?? 0; ?>
                                                    </h5>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-gradient-warning">
                                            <div class="card-body p-3">
                                                <div class="numbers">
                                                    <p class="text-white text-sm mb-0">Average Payment</p>
                                                    <h5 class="text-white font-weight-bolder mb-0">
                                                        <?php echo formatMoney($summary['average_payment'] ?? 0); ?>
                                                    </h5>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>

                    <div class="row">
                        <!-- Daily Collections Chart -->
                        <div class="col-md-8">
                            <div class="card mb-4">
                                <div class="card-header pb-0">
                                    <h6>Daily Collections Trend</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="dailyTrendChart"></canvas>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Methods -->
                        <div class="col-md-4">
                            <div class="card mb-4">
                                <div class="card-header pb-0">
                                    <h6>Payment Methods</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="methodsChart"></canvas>
                                    <div class="mt-3">
                                        <?php while ($method = $payment_methods->fetch_assoc()): ?>
                                            <div class="d-flex justify-content-between align-items-center mb-2">
                                                <span><?php echo htmlspecialchars($method['payment_method']); ?></span>
                                                <span>
                                                    <?php echo formatMoney($method['total']); ?>
                                                    <small class="text-secondary">(<?php echo $method['count']; ?>)</small>
                                                </span>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Program Breakdown -->
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0" id="programTable">
                            <thead>
                                <tr>
                                    <th>Program</th>
                                    <th>Students</th>
                                    <th>Total Collections</th>
                                    <th>% of Total</th>
                                    <th class="no-export">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                while ($program = $programs->fetch_assoc()): 
                                    $percentage = $summary['total_amount'] > 0 
                                        ? ($program['total_amount'] / $summary['total_amount']) * 100 
                                        : 0;
                                ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($program['program_code']); ?></td>
                                    <td><?php echo $program['student_count']; ?></td>
                                    <td><?php echo formatMoney($program['total_amount']); ?></td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <span class="me-2"><?php echo round($percentage, 1); ?>%</span>
                                            <div class="progress" style="width: 100px">
                                                <div class="progress-bar bg-gradient-info" 
                                                     role="progressbar" 
                                                     style="width: <?php echo $percentage; ?>%"></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="no-export">
                                        <a href="daily_collections.php?program=<?php echo urlencode($program['program_code']); ?>&month=<?php echo $month; ?>&year=<?php echo $year; ?>" 
                                           class="btn btn-info btn-sm">
                                            View Details
                                        </a>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print, form {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
    th.no-export, td.no-export {
        display: none;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
function updateMonthYear(value) {
    const [year, month] = value.split('-');
    window.location.href = `?month=${month}&year=${year}`;
}

function exportToExcel() {
    let table = document.getElementById("programTable");
    let html = table.outerHTML;
    
    let blob = new Blob([html], { type: 'application/vnd.ms-excel' });
    let url = window.URL.createObjectURL(blob);
    
    let a = document.createElement("a");
    a.href = url;
    a.download = 'monthly_summary_<?php echo sprintf('%s_%s', $year, $month); ?>.xls';
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
}

document.addEventListener('DOMContentLoaded', function() {
    // Daily Collections Trend Chart
    const dailyData = <?php 
        $chartData = [];
        while ($row = $collections->fetch_assoc()) {
            $chartData[] = [
                'date' => date('j', strtotime($row['payment_date'])),
                'amount' => $row['daily_total'],
                'transactions' => $row['transactions']
            ];
        }
        echo json_encode($chartData);
    ?>;

    new Chart(document.getElementById('dailyTrendChart'), {
        type: 'bar',
        data: {
            labels: dailyData.map(row => row.date),
            datasets: [{
                label: 'Daily Collections',
                data: dailyData.map(row => row.amount),
                backgroundColor: 'rgba(66, 135, 245, 0.5)',
                borderColor: 'rgb(66, 135, 245)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true
                }
            }
        }
    });

    // Payment Methods Chart
    const methodsData = <?php 
        $conn->data_seek(0);
        $methodsChartData = [];
        while ($row = $payment_methods->fetch_assoc()) {
            $methodsChartData[] = [
                'method' => $row['payment_method'],
                'total' => $row['total']
            ];
        }
        echo json_encode($methodsChartData);
    ?>;

    new Chart(document.getElementById('methodsChart'), {
        type: 'doughnut',
        data: {
            labels: methodsData.map(row => row.method),
            datasets: [{
                data: methodsData.map(row => row.total),
                backgroundColor: [
                    'rgba(66, 135, 245, 0.5)',
                    'rgba(245, 171, 66, 0.5)',
                    'rgba(66, 245, 108, 0.5)',
                    'rgba(245, 66, 66, 0.5)'
                ]
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>