<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access only
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get date range parameters
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Get payment analysis data
$analysis_query = "
    SELECT 
        DATE(p.payment_date) as date,
        COUNT(*) as transactions,
        SUM(p.amount) as total_amount,
        COUNT(DISTINCT sf.student_id) as unique_students,
        AVG(p.amount) as average_amount,
        MIN(p.amount) as min_amount,
        MAX(p.amount) as max_amount
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE p.payment_date BETWEEN ? AND ?
    AND p.status = 'confirmed'
    GROUP BY DATE(p.payment_date)
    ORDER BY date
";

$stmt = $conn->prepare($analysis_query);
$stmt->bind_param('ss', $start_date, $end_date);
$stmt->execute();
$daily_stats = $stmt->get_result();

// Get payment methods trend
$methods_trend = "
    SELECT 
        DATE(p.payment_date) as date,
        p.payment_method,
        COUNT(*) as count,
        SUM(p.amount) as total
    FROM payments p
    WHERE p.payment_date BETWEEN ? AND ?
    AND p.status = 'confirmed'
    GROUP BY DATE(p.payment_date), p.payment_method
    ORDER BY date, p.payment_method
";

$stmt = $conn->prepare($methods_trend);
$stmt->bind_param('ss', $start_date, $end_date);
$stmt->execute();
$methods_data = $stmt->get_result();

// Get overall summary
$summary_query = "
    SELECT 
        COUNT(*) as total_transactions,
        SUM(p.amount) as total_amount,
        COUNT(DISTINCT sf.student_id) as total_students,
        AVG(p.amount) as average_payment,
        MIN(p.amount) as min_payment,
        MAX(p.amount) as max_payment
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE p.payment_date BETWEEN ? AND ?
    AND p.status = 'confirmed'
";

$stmt = $conn->prepare($summary_query);
$stmt->bind_param('ss', $start_date, $end_date);
$stmt->execute();
$summary = $stmt->get_result()->fetch_assoc();

// Calculate daily averages
$days_diff = (strtotime($end_date) - strtotime($start_date)) / (60 * 60 * 24);
$daily_average = $summary['total_amount'] / ($days_diff + 1);

$page_title = "Payment Analysis";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Payment Analysis</h6>
                        <div>
                            <button onclick="exportToExcel()" class="btn btn-success btn-sm me-2">
                                <i class="fas fa-file-excel"></i> Export
                            </button>
                            <button onclick="window.print()" class="btn btn-info btn-sm">
                                <i class="fas fa-print"></i> Print
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Date Range Selector -->
                    <form method="GET" class="mb-4">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>Start Date</label>
                                    <input type="date" name="start_date" class="form-control" 
                                           value="<?php echo $start_date; ?>">
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>End Date</label>
                                    <input type="date" name="end_date" class="form-control" 
                                           value="<?php echo $end_date; ?>">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary d-block">
                                        Apply Filter
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>

                    <!-- Summary Statistics -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <div class="card bg-gradient-primary">
                                <div class="card-body p-3">
                                    <div class="numbers">
                                        <p class="text-white text-sm mb-0">Total Collections</p>
                                        <h5 class="text-white font-weight-bolder mb-0">
                                            <?php echo formatMoney($summary['total_amount'] ?? 0); ?>
                                        </h5>
                                        <small class="text-white">
                                            Daily Avg: <?php echo formatMoney($daily_average); ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-success">
                                <div class="card-body p-3">
                                    <div class="numbers">
                                        <p class="text-white text-sm mb-0">Total Transactions</p>
                                        <h5 class="text-white font-weight-bolder mb-0">
                                            <?php echo number_format($summary['total_transactions']); ?>
                                        </h5>
                                        <small class="text-white">
                                            Daily Avg: <?php echo round($summary['total_transactions'] / ($days_diff + 1), 1); ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-info">
                                <div class="card-body p-3">
                                    <div class="numbers">
                                        <p class="text-white text-sm mb-0">Average Payment</p>
                                        <h5 class="text-white font-weight-bolder mb-0">
                                            <?php echo formatMoney($summary['average_payment'] ?? 0); ?>
                                        </h5>
                                        <small class="text-white">
                                            Range: <?php echo formatMoney($summary['min_payment']); ?> - <?php echo formatMoney($summary['max_payment']); ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card bg-gradient-warning">
                                <div class="card-body p-3">
                                    <div class="numbers">
                                        <p class="text-white text-sm mb-0">Unique Students</p>
                                        <h5 class="text-white font-weight-bolder mb-0">
                                            <?php echo number_format($summary['total_students']); ?>
                                        </h5>
                                        <small class="text-white">
                                            Avg per day: <?php echo round($summary['total_students'] / ($days_diff + 1), 1); ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Charts -->
                    <div class="row mb-4">
                        <div class="col-md-8">
                            <div class="card">
                                <div class="card-header pb-0">
                                    <h6>Daily Collections Trend</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="trendChart"></canvas>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-header pb-0">
                                    <h6>Payment Methods Distribution</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="methodsChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Daily Statistics Table -->
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0" id="statsTable">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Transactions</th>
                                    <th>Total Amount</th>
                                    <th>Average Payment</th>
                                    <th>Students</th>
                                    <th>Min Payment</th>
                                    <th>Max Payment</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($day = $daily_stats->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo date('M j, Y', strtotime($day['date'])); ?></td>
                                    <td><?php echo $day['transactions']; ?></td>
                                    <td><?php echo formatMoney($day['total_amount']); ?></td>
                                    <td><?php echo formatMoney($day['average_amount']); ?></td>
                                    <td><?php echo $day['unique_students']; ?></td>
                                    <td><?php echo formatMoney($day['min_amount']); ?></td>
                                    <td><?php echo formatMoney($day['max_amount']); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print, form {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
function exportToExcel() {
    let table = document.getElementById("statsTable");
    let html = table.outerHTML;
    
    let blob = new Blob([html], { type: 'application/vnd.ms-excel' });
    let url = window.URL.createObjectURL(blob);
    
    let a = document.createElement("a");
    a.href = url;
    a.download = 'payment_analysis_<?php echo $start_date; ?>_to_<?php echo $end_date; ?>.xls';
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
}

document.addEventListener('DOMContentLoaded', function() {
    // Daily Trend Chart
    const trendData = <?php 
        $conn->data_seek(0);
        $trendChartData = [];
        while ($row = $daily_stats->fetch_assoc()) {
            $trendChartData[] = [
                'date' => date('M j', strtotime($row['date'])),
                'amount' => $row['total_amount'],
                'transactions' => $row['transactions']
            ];
        }
        echo json_encode($trendChartData);
    ?>;

    new Chart(document.getElementById('trendChart'), {
        type: 'line',
        data: {
            labels: trendData.map(row => row.date),
            datasets: [{
                label: 'Daily Collections',
                data: trendData.map(row => row.amount),
                borderColor: 'rgb(66, 135, 245)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true
                }
            }
        }
    });

    // Payment Methods Chart
    const methodsData = <?php 
        $methods_summary = [];
        while ($row = $methods_data->fetch_assoc()) {
            if (!isset($methods_summary[$row['payment_method']])) {
                $methods_summary[$row['payment_method']] = [
                    'total' => 0,
                    'count' => 0
                ];
            }
            $methods_summary[$row['payment_method']]['total'] += $row['total'];
            $methods_summary[$row['payment_method']]['count'] += $row['count'];
        }
        echo json_encode($methods_summary);
    ?>;

    new Chart(document.getElementById('methodsChart'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(methodsData),
            datasets: [{
                data: Object.values(methodsData).map(m => m.total),
                backgroundColor: [
                    'rgba(66, 135, 245, 0.5)',
                    'rgba(245, 171, 66, 0.5)',
                    'rgba(66, 245, 108, 0.5)',
                    'rgba(245, 66, 66, 0.5)'
                ]
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>