<?php
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/payment_logger.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get filter parameters
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$status = $_GET['status'] ?? '';
$reference = $_GET['reference'] ?? '';

// Get payment logs
$payment_logs = getPaymentLogs($reference, $status, $start_date, $end_date);

$page_title = "Payment Monitoring";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Payment Monitoring Dashboard</h5>
                    <button type="button" class="btn btn-light btn-sm" onclick="refreshPayments()">
                        <i class="fas fa-sync-alt"></i> Refresh
                    </button>
                </div>
                <div class="card-body">
                    <!-- Filters -->
                    <form method="GET" class="row g-3 mb-4">
                        <div class="col-md-3">
                            <label class="form-label">Start Date</label>
                            <input type="date" name="start_date" class="form-control" value="<?php echo $start_date; ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">End Date</label>
                            <input type="date" name="end_date" class="form-control" value="<?php echo $end_date; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Status</label>
                            <select name="status" class="form-select">
                                <option value="">All</option>
                                <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="success" <?php echo $status === 'success' ? 'selected' : ''; ?>>Success</option>
                                <option value="failed" <?php echo $status === 'failed' ? 'selected' : ''; ?>>Failed</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Reference Number</label>
                            <input type="text" name="reference" class="form-control" value="<?php echo htmlspecialchars($reference); ?>">
                        </div>
                        <div class="col-md-1">
                            <label class="form-label">&nbsp;</label>
                            <button type="submit" class="btn btn-primary d-block">Filter</button>
                        </div>
                    </form>

                    <!-- Payment Logs Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Student</th>
                                    <th>Reference</th>
                                    <th>Fee Type</th>
                                    <th>Amount</th>
                                    <th>Method</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($log = $payment_logs->fetch_assoc()): ?>
                                    <tr>
                                        <td><?php echo date('Y-m-d H:i:s', strtotime($log['created_at'])); ?></td>
                                        <td>
                                            <?php if ($log['student_number']): ?>
                                                <?php echo htmlspecialchars($log['student_number']); ?><br>
                                                <small><?php echo htmlspecialchars($log['first_name'] . ' ' . $log['last_name']); ?></small>
                                            <?php else: ?>
                                                <span class="text-muted">N/A</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($log['reference_number']); ?></td>
                                        <td><?php echo htmlspecialchars($log['fee_type'] ?? 'N/A'); ?></td>
                                        <td><?php echo formatMoney($log['amount']); ?></td>
                                        <td><?php echo ucfirst($log['payment_method']); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo getStatusBadgeClass($log['status']); ?>">
                                                <?php echo ucfirst($log['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button type="button" 
                                                    class="btn btn-sm btn-info"
                                                    onclick="viewDetails('<?php echo $log['reference_number']; ?>')">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            <?php if ($log['status'] === 'pending'): ?>
                                                <button type="button" 
                                                        class="btn btn-sm btn-success"
                                                        onclick="verifyPayment('<?php echo $log['reference_number']; ?>')">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Payment Details Modal -->
<div class="modal fade" id="paymentDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Payment Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="paymentDetailsContent">
                <!-- Content will be loaded dynamically -->
            </div>
        </div>
    </div>
</div>

<script>
function getStatusBadgeClass(status) {
    switch(status.toLowerCase()) {
        case 'success':
        case 'confirmed':
            return 'success';
        case 'pending':
            return 'warning';
        case 'failed':
        case 'rejected':
            return 'danger';
        default:
            return 'secondary';
    }
}

function refreshPayments() {
    location.reload();
}

function viewDetails(reference) {
    const modal = new bootstrap.Modal(document.getElementById('paymentDetailsModal'));
    
    // Fetch payment details
    fetch(`get_payment_details.php?reference=${reference}`)
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                document.getElementById('paymentDetailsContent').innerHTML = data.html;
                modal.show();
            } else {
                alert('Error loading payment details: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading payment details');
        });
}

function verifyPayment(reference) {
    if (!confirm('Are you sure you want to verify this payment?')) {
        return;
    }
    
    fetch(`verify_payment.php?reference=${reference}`)
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                alert('Payment verified successfully!');
                location.reload();
            } else {
                alert('Error verifying payment: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error verifying payment');
        });
}

// Auto-refresh every 5 minutes
setInterval(refreshPayments, 300000);
</script>

<?php
function getStatusBadgeClass($status) {
    switch(strtolower($status)) {
        case 'success':
        case 'confirmed':
            return 'success';
        case 'pending':
            return 'warning';
        case 'failed':
        case 'rejected':
            return 'danger';
        default:
            return 'secondary';
    }
}

require_once '../includes/footer.php';
?>