<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $student_number = trim($_POST['student_number'] ?? '');
    $fee_type_id = intval($_POST['fee_type_id'] ?? 0);
    $amount = floatval($_POST['amount'] ?? 0);
    $payment_method = trim($_POST['payment_method'] ?? '');
    $reference_number = trim($_POST['reference_number'] ?? '');
    $payment_date = trim($_POST['payment_date'] ?? '');
    $remarks = trim($_POST['remarks'] ?? '');

    $errors = [];

    // Validation
    if (empty($student_number)) {
        $errors[] = "Student number is required.";
    }
    if ($fee_type_id <= 0) {
        $errors[] = "Please select a fee type.";
    }
    if ($amount <= 0) {
        $errors[] = "Amount must be greater than 0.";
    }
    if (empty($payment_method)) {
        $errors[] = "Payment method is required.";
    }
    if (empty($reference_number)) {
        $errors[] = "Reference number is required.";
    }
    if (empty($payment_date)) {
        $errors[] = "Payment date is required.";
    }

    if (empty($errors)) {
        // Get student ID from student number
        $stmt = $conn->prepare("SELECT id FROM students WHERE student_number = ?");
        $stmt->bind_param("s", $student_number);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($student = $result->fetch_assoc()) {
            $student_id = $student['id'];
            
            // Get or create student fee record
            $stmt = $conn->prepare("
                SELECT id FROM student_fees 
                WHERE student_id = ? AND fee_type_id = ?
            ");
            $stmt->bind_param("ii", $student_id, $fee_type_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($student_fee = $result->fetch_assoc()) {
                $student_fee_id = $student_fee['id'];
            } else {
                // Create new student fee record
                $stmt = $conn->prepare("
                    INSERT INTO student_fees (student_id, fee_type_id) 
                    VALUES (?, ?)
                ");
                $stmt->bind_param("ii", $student_id, $fee_type_id);
                $stmt->execute();
                $student_fee_id = $conn->insert_id;
            }

            // Record the payment
            $stmt = $conn->prepare("
                INSERT INTO payments (
                    student_fee_id, amount, payment_method, reference_number, 
                    payment_date, status, remarks
                ) VALUES (?, ?, ?, ?, ?, 'confirmed', ?)
            ");
            $stmt->bind_param(
                "idssss",
                $student_fee_id, $amount, $payment_method, $reference_number,
                $payment_date, $remarks
            );

            if ($stmt->execute()) {
                $_SESSION['success_message'] = '<i class="fas fa-check-circle text-success"></i> Manual payment recorded successfully.';
                header("Location: dashboard.php");
                exit();
            } else {
                $errors[] = "Failed to record payment. Please try again.";
            }
        } else {
            $errors[] = "Student not found with the provided student number.";
        }
    }

    if (!empty($errors)) {
        $_SESSION['error_message'] = implode("<br>", $errors);
    }
}

// Get fee types
$fee_types_query = "SELECT id, name, amount FROM fee_types WHERE status = 'active' ORDER BY name";
$fee_types = $conn->query($fee_types_query);

$page_title = "Record Manual Payment";
require_once '../includes/header.php';
?>

<div class="container py-4">
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Record Manual Payment</h5>
                        <a href="dashboard.php" class="btn btn-secondary btn-sm">
                            <i class="fas fa-arrow-left"></i> Back to Dashboard
                        </a>
                    </div>
                    <p class="text-muted small mb-0 mt-2">
                        Use this form to record payments made through bank transfer, cash, or mobile money.
                        For online payments, students should use the payment portal.
                    </p>
                </div>
                <div class="card-body">
                    <form method="post" class="needs-validation" novalidate>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="student_number" class="form-label">Student Number <span class="text-danger">*</span></label>
                                <input type="text" name="student_number" id="student_number" 
                                       class="form-control" required
                                       value="<?php echo htmlspecialchars($_POST['student_number'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6">
                                <label for="fee_type_id" class="form-label">Fee Type <span class="text-danger">*</span></label>
                                <select name="fee_type_id" id="fee_type_id" class="form-select" required>
                                    <option value="">Select Fee Type</option>
                                    <?php while($fee = $fee_types->fetch_assoc()): ?>
                                        <option value="<?php echo $fee['id']; ?>" 
                                                data-amount="<?php echo $fee['amount']; ?>"
                                                <?php echo (isset($_POST['fee_type_id']) && $_POST['fee_type_id'] == $fee['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($fee['name']); ?> 
                                            (GHs <?php echo number_format($fee['amount'], 2); ?>)
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="amount" class="form-label">Amount (GHs) <span class="text-danger">*</span></label>
                                <input type="number" name="amount" id="amount" class="form-control" 
                                       step="0.01" required
                                       value="<?php echo htmlspecialchars($_POST['amount'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6">
                                <label for="payment_method" class="form-label">Payment Method <span class="text-danger">*</span></label>
                                <select name="payment_method" id="payment_method" class="form-select" required>
                                    <option value="">Select Payment Method</option>
                                    <option value="cash" <?php echo (isset($_POST['payment_method']) && $_POST['payment_method'] == 'cash') ? 'selected' : ''; ?>>Cash</option>
                                    <option value="bank_transfer" <?php echo (isset($_POST['payment_method']) && $_POST['payment_method'] == 'bank_transfer') ? 'selected' : ''; ?>>Bank Transfer</option>
                                    <option value="momo" <?php echo (isset($_POST['payment_method']) && $_POST['payment_method'] == 'momo') ? 'selected' : ''; ?>>Mobile Money</option>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="reference_number" class="form-label">Reference Number <span class="text-danger">*</span></label>
                                <input type="text" name="reference_number" id="reference_number" 
                                       class="form-control" required
                                       value="<?php echo htmlspecialchars($_POST['reference_number'] ?? ''); ?>">
                                <div class="form-text">Bank transfer reference or receipt number</div>
                            </div>
                            <div class="col-md-6">
                                <label for="payment_date" class="form-label">Payment Date <span class="text-danger">*</span></label>
                                <input type="date" name="payment_date" id="payment_date" 
                                       class="form-control" required
                                       max="<?php echo date('Y-m-d'); ?>"
                                       value="<?php echo htmlspecialchars($_POST['payment_date'] ?? date('Y-m-d')); ?>">
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="remarks" class="form-label">Remarks</label>
                            <textarea name="remarks" id="remarks" class="form-control" rows="2"
                                    placeholder="Any additional notes about this payment"><?php echo htmlspecialchars($_POST['remarks'] ?? ''); ?></textarea>
                        </div>

                        <div class="d-flex justify-content-end gap-2">
                            <button type="reset" class="btn btn-secondary">
                                <i class="fas fa-undo"></i> Reset
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> Record Payment
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Auto-fill amount when fee type is selected
document.getElementById('fee_type_id').addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    if (selectedOption.value) {
        const amount = selectedOption.getAttribute('data-amount');
        document.getElementById('amount').value = amount;
    }
});

// Form validation
(function () {
    'use strict'
    var forms = document.querySelectorAll('.needs-validation')
    Array.prototype.slice.call(forms)
        .forEach(function (form) {
            form.addEventListener('submit', function (event) {
                if (!form.checkValidity()) {
                    event.preventDefault()
                    event.stopPropagation()
                }
                form.classList.add('was-validated')
            }, false)
        })
})()
</script>

<?php require_once '../includes/footer.php'; ?>