<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access only
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

$page_title = "Financial Reports";
require_once '../includes/header.php';

// Get quick statistics
$current_academic_year = getCurrentAcademicYear($conn);

// Total collections for current academic year
$collections_query = "
    SELECT 
        SUM(p.amount) as total_collected,
        COUNT(DISTINCT p.id) as total_transactions
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    WHERE p.status = 'confirmed'
    AND sf.academic_year_id = ?
";
$stmt = $conn->prepare($collections_query);
$stmt->bind_param('i', $current_academic_year);
$stmt->execute();
$collections = $stmt->get_result()->fetch_assoc();

// Get program-wise collections
$program_collections = "
    SELECT 
        p.program_code,
        p.program_name,
        COUNT(DISTINCT s.id) as total_students,
        SUM(sf.amount) as total_fees,
        SUM(CASE WHEN py.status = 'confirmed' THEN py.amount ELSE 0 END) as total_paid
    FROM programs p
    LEFT JOIN students s ON s.program_id = p.id
    LEFT JOIN student_fees sf ON s.id = sf.student_id
    LEFT JOIN payments py ON sf.id = py.student_fee_id
    WHERE sf.academic_year_id = ?
    GROUP BY p.id
    ORDER BY p.program_code
";
$stmt = $conn->prepare($program_collections);
$stmt->bind_param('i', $current_academic_year);
$stmt->execute();
$programs = $stmt->get_result();

// Get monthly collections trend
$monthly_trend = "
    SELECT 
        DATE_FORMAT(payment_date, '%Y-%m') as month,
        COUNT(*) as transaction_count,
        SUM(amount) as total_amount
    FROM payments
    WHERE status = 'confirmed'
    AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
    GROUP BY DATE_FORMAT(payment_date, '%Y-%m')
    ORDER BY month DESC
";
$monthly_data = $conn->query($monthly_trend);

// Get payment methods distribution
$payment_methods = "
    SELECT 
        payment_method,
        COUNT(*) as count,
        SUM(amount) as total
    FROM payments
    WHERE status = 'confirmed'
    AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY payment_method
";
$methods_data = $conn->query($payment_methods);
?>

<div class="container-fluid py-4">
    <!-- Reports Navigation -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0">Available Reports</h5>
                        <div>
                            <a href="batch_report.php" class="btn btn-primary btn-sm">
                                <i class="fas fa-users"></i> Batch Report
                            </a>
                            <button onclick="window.print()" class="btn btn-info btn-sm ms-2">
                                <i class="fas fa-print"></i> Print Current
                            </button>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <div class="card bg-gradient-primary">
                                <div class="card-body p-3">
                                    <div class="row">
                                        <div class="col-8">
                                            <div class="numbers">
                                                <p class="text-white text-sm mb-0 text-capitalize font-weight-bold">Collections This Year</p>
                                                <h5 class="text-white font-weight-bolder mb-0">
                                                    <?php echo formatMoney($collections['total_collected'] ?? 0); ?>
                                                </h5>
                                                <small class="text-white"><?php echo number_format($collections['total_transactions']); ?> transactions</small>
                                            </div>
                                        </div>
                                        <div class="col-4 text-end">
                                            <div class="icon icon-shape bg-white shadow text-center border-radius-md">
                                                <i class="fas fa-money-bill-wave text-primary opacity-10"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-8">
                            <div class="card">
                                <div class="card-body">
                                    <h6>Quick Reports</h6>
                                    <div class="row">
                                        <div class="col-md-4">
                                            <a href="daily_collections.php" class="btn btn-outline-primary w-100 mb-2">
                                                <i class="fas fa-calendar-day"></i> Daily Collections
                                            </a>
                                        </div>
                                        <div class="col-md-4">
                                            <a href="monthly_summary.php" class="btn btn-outline-info w-100 mb-2">
                                                <i class="fas fa-calendar-alt"></i> Monthly Summary
                                            </a>
                                        </div>
                                        <div class="col-md-4">
                                            <a href="payment_analysis.php" class="btn btn-outline-success w-100 mb-2">
                                                <i class="fas fa-chart-line"></i> Payment Analysis
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Program-wise Collections -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header pb-0">
                    <h6>Program-wise Collections</h6>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th>Program</th>
                                    <th>Students</th>
                                    <th>Total Fees</th>
                                    <th>Collected</th>
                                    <th>Collection Rate</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($program = $programs->fetch_assoc()): 
                                    $collection_rate = $program['total_fees'] > 0 
                                        ? ($program['total_paid'] / $program['total_fees']) * 100 
                                        : 0;
                                ?>
                                    <tr>
                                        <td>
                                            <h6 class="mb-0 text-sm"><?php echo htmlspecialchars($program['program_code']); ?></h6>
                                            <p class="text-xs text-secondary mb-0"><?php echo htmlspecialchars($program['program_name']); ?></p>
                                        </td>
                                        <td><?php echo $program['total_students']; ?></td>
                                        <td><?php echo formatMoney($program['total_fees']); ?></td>
                                        <td><?php echo formatMoney($program['total_paid']); ?></td>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <span class="me-2"><?php echo round($collection_rate, 1); ?>%</span>
                                                <div class="progress" style="width: 100px">
                                                    <div class="progress-bar bg-gradient-success" 
                                                         role="progressbar" 
                                                         style="width: <?php echo $collection_rate; ?>%"></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <a href="batch_report.php?course=<?php echo urlencode($program['program_code']); ?>" 
                                               class="btn btn-info btn-sm">
                                                View Details
                                            </a>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Collections Trend -->
    <div class="row mb-4">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header pb-0">
                    <h6>Monthly Collections Trend</h6>
                </div>
                <div class="card-body">
                    <canvas id="monthlyTrendChart" class="chart-canvas"></canvas>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card">
                <div class="card-header pb-0">
                    <h6>Payment Methods (Last 30 Days)</h6>
                </div>
                <div class="card-body">
                    <canvas id="paymentMethodsChart"></canvas>
                    <div class="mt-3">
                        <?php while ($method = $methods_data->fetch_assoc()): ?>
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span class="text-sm"><?php echo htmlspecialchars($method['payment_method']); ?></span>
                                <span class="text-sm font-weight-bold">
                                    <?php echo formatMoney($method['total']); ?>
                                    <small class="text-secondary">(<?php echo $method['count']; ?>)</small>
                                </span>
                            </div>
                        <?php endwhile; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Monthly Trend Chart
    const monthlyData = <?php 
        $chartData = [];
        while ($row = $monthly_data->fetch_assoc()) {
            $chartData[] = [
                'month' => date('M Y', strtotime($row['month'] . '-01')),
                'amount' => $row['total_amount'],
                'count' => $row['transaction_count']
            ];
        }
        echo json_encode(array_reverse($chartData));
    ?>;

    new Chart(document.getElementById('monthlyTrendChart'), {
        type: 'bar',
        data: {
            labels: monthlyData.map(row => row.month),
            datasets: [{
                label: 'Collections',
                data: monthlyData.map(row => row.amount),
                backgroundColor: 'rgba(66, 135, 245, 0.5)',
                borderColor: 'rgb(66, 135, 245)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true
                }
            }
        }
    });

    // Payment Methods Chart
    const methodsData = <?php 
        $conn->data_seek(0);
        $methodsChartData = [];
        while ($row = $methods_data->fetch_assoc()) {
            $methodsChartData[] = [
                'method' => $row['payment_method'],
                'total' => $row['total']
            ];
        }
        echo json_encode($methodsChartData);
    ?>;

    new Chart(document.getElementById('paymentMethodsChart'), {
        type: 'doughnut',
        data: {
            labels: methodsData.map(row => row.method),
            datasets: [{
                data: methodsData.map(row => row.total),
                backgroundColor: [
                    'rgba(66, 135, 245, 0.5)',
                    'rgba(245, 171, 66, 0.5)',
                    'rgba(66, 245, 108, 0.5)',
                    'rgba(245, 66, 66, 0.5)'
                ]
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>