<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify accountant access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit();
}

// Get payment ID
$payment_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($payment_id <= 0) {
    $_SESSION['error_message'] = "Invalid payment ID.";
    header('Location: dashboard.php');
    exit();
}

// Handle payment confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $comment = trim($_POST['comment'] ?? '');
    $updated_by = $_SESSION['user_id'];
    
    if ($action === 'confirm') {
        // Update payment status to confirmed
        $stmt = $conn->prepare("UPDATE payments SET 
            status = 'confirmed', 
            updated_at = NOW(), 
            updated_by = ?,
            review_comment = ?,
            confirmation_date = NOW()
            WHERE id = ?");
        $stmt->bind_param("isi", $updated_by, $comment, $payment_id);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = '<i class="fas fa-check-circle text-success"></i> Payment verified and confirmed successfully.';
        } else {
            $_SESSION['error_message'] = "Failed to confirm payment.";
        }
        $stmt->close();
    } elseif ($action === 'reject') {
        if (empty($comment)) {
            $_SESSION['error_message'] = "Please provide a reason for rejecting the payment.";
            header("Location: review_payment.php?id=" . $payment_id);
            exit();
        }
        // Update payment status to rejected
        $stmt = $conn->prepare("UPDATE payments SET 
            status = 'rejected', 
            updated_at = NOW(), 
            updated_by = ?,
            review_comment = ?,
            rejection_date = NOW()
            WHERE id = ?");
        $stmt->bind_param("isi", $updated_by, $comment, $payment_id);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = '<i class="fas fa-times-circle text-danger"></i> Payment has been rejected. Student will be notified.';
        } else {
            $_SESSION['error_message'] = "Failed to reject payment.";
        }
        $stmt->close();
    }
    
    header('Location: dashboard.php');
    exit();
}

// Get payment details
$stmt = $conn->prepare("
    SELECT 
        p.*,
        s.student_number,
        s.first_name,
        s.last_name,
        ft.name as fee_type,
        ft.amount as total_fee_amount,
        (
            SELECT SUM(amount) 
            FROM payments 
            WHERE student_fee_id = sf.id 
            AND status = 'confirmed'
        ) as total_paid_amount
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    WHERE p.id = ?
");

$stmt->bind_param("i", $payment_id);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$payment) {
    $_SESSION['error_message'] = "Payment not found.";
    header('Location: dashboard.php');
    exit();
}

$page_title = "Review Payment";
require_once '../includes/header.php';
?>

<div class="container py-4">
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">Payment Details</h5>
                    <a href="dashboard.php" class="btn btn-secondary btn-sm">
                        <i class="fas fa-arrow-left"></i> Back
                    </a>
                </div>
                <div class="card-body">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <h6>Student Information</h6>
                            <p class="mb-1">
                                <strong>Name:</strong> <?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?>
                            </p>
                            <p class="mb-1">
                                <strong>Student ID:</strong> <?php echo htmlspecialchars($payment['student_number']); ?>
                            </p>
                        </div>
                        <div class="col-md-6">
                            <h6>Payment Information</h6>
                            <p class="mb-1">
                                <strong>Amount:</strong> GHs <?php echo number_format($payment['amount'], 2); ?>
                            </p>
                            <p class="mb-1">
                                <strong>Reference:</strong> <?php echo htmlspecialchars($payment['reference_number']); ?>
                            </p>
                            <p class="mb-1">
                                <strong>Status:</strong> 
                                <span class="badge bg-<?php echo $payment['status'] === 'confirmed' ? 'success' : ($payment['status'] === 'pending' ? 'warning' : 'danger'); ?>">
                                    <?php echo ucfirst($payment['status']); ?>
                                </span>
                            </p>
                            <p class="mb-1">
                                <strong>Date:</strong> <?php echo date('M d, Y g:i A', strtotime($payment['created_at'])); ?>
                            </p>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-12">
                            <h6>Fee Details</h6>
                            <p class="mb-1">
                                <strong>Fee Type:</strong> <?php echo htmlspecialchars($payment['fee_type']); ?>
                            </p>
                            <p class="mb-1">
                                <strong>Total Fee Amount:</strong> GHs <?php echo number_format($payment['total_fee_amount'], 2); ?>
                            </p>
                            <p class="mb-1">
                                <strong>Total Paid:</strong> GHs <?php echo number_format($payment['total_paid_amount'], 2); ?>
                            </p>
                        </div>
                    </div>

                    <?php if ($payment['status'] === 'pending'): ?>
                    <div class="row">
                        <div class="col-12">
                            <hr>
                            <form method="post">
                                <div class="mb-3">
                                    <label for="comment" class="form-label">Review Comment</label>
                                    <textarea name="comment" id="comment" class="form-control" rows="3" 
                                              placeholder="Enter verification notes or rejection reason"></textarea>
                                    <div class="form-text">Required for rejections, optional for confirmations.</div>
                                </div>
                                <div class="d-flex justify-content-end gap-2">
                                    <button type="submit" name="action" value="reject" class="btn btn-danger" 
                                            onclick="return validateReject(this.form);">
                                        <i class="fas fa-times"></i> Reject Payment
                                    </button>
                                    <button type="submit" name="action" value="confirm" class="btn btn-success" 
                                            onclick="return confirm('Are you sure you want to confirm this payment?')">
                                        <i class="fas fa-check"></i> Verify & Confirm
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    <script>
                    function validateReject(form) {
                        if (form.comment.value.trim() === '') {
                            alert('Please provide a reason for rejecting the payment.');
                            form.comment.focus();
                            return false;
                        }
                        return confirm('Are you sure you want to reject this payment?');
                    }
                    </script>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>