<?php
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/date_functions.php';
require_once '../includes/payment_functions.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Get summary statistics
$stats = [
    'total_students' => 0,
    'total_payments' => 0,
    'pending_payments' => 0,
    'total_collected' => 0
];

try {
    // Get total students
    $result = $conn->query("SELECT COUNT(*) as count FROM students");
    if ($result) {
        $stats['total_students'] = $result->fetch_assoc()['count'];
    }
} catch (Exception $e) {
    error_log("Error getting total students: " . $e->getMessage());
}

// Get payment statistics
try {
    $result = $conn->query("
        SELECT 
            COUNT(*) as total_payments,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_payments,
            SUM(CASE WHEN status = 'confirmed' THEN amount ELSE 0 END) as total_collected,
            SUM(CASE WHEN status = 'refunded' THEN 1 ELSE 0 END) as refunded_payments,
            SUM(CASE WHEN status = 'refunded' THEN refunded_amount ELSE 0 END) as total_refunded
        FROM payments
    ");
    
    if ($result) {
        $payment_stats = $result->fetch_assoc();
        $stats['total_payments'] = $payment_stats['total_payments'] ?? 0;
        $stats['pending_payments'] = $payment_stats['pending_payments'] ?? 0;
        $stats['total_collected'] = $payment_stats['total_collected'] ?? 0;
        $stats['refunded_payments'] = $payment_stats['refunded_payments'] ?? 0;
        $stats['total_refunded'] = $payment_stats['total_refunded'] ?? 0;
    }
} catch (Exception $e) {
    error_log("Error getting payment statistics: " . $e->getMessage());
}

// Get sorting parameters
$sort_column = isset($_GET['sort']) ? $_GET['sort'] : 'created_at';
$sort_order = isset($_GET['order']) ? $_GET['order'] : 'DESC';

// Get filter parameters
$program_filter = isset($_GET['program']) ? (int)$_GET['program'] : 0;
$year_level_filter = isset($_GET['year_level']) ? (int)$_GET['year_level'] : 0;

// Validate sort column to prevent SQL injection
$allowed_sort_columns = [
    'student_number' => 's.student_number',
    'name' => 's.first_name',
    'program' => 'p.program_name',
    'created_at' => 'u.created_at'
];

$sort_by = $allowed_sort_columns[$sort_column] ?? 'u.created_at';
$sort_direction = $sort_order === 'ASC' ? 'ASC' : 'DESC';

// Get all programs for filter dropdown
$programs_result = $conn->query("SELECT id, program_name FROM programs ORDER BY program_name");
$programs = [];
while ($program = $programs_result->fetch_assoc()) {
    $programs[] = $program;
}

// Pagination settings
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$items_per_page = 10;
$offset = ($page - 1) * $items_per_page;

// Pagination settings for both tables
$registrations_page = isset($_GET['reg_page']) ? max(1, (int)$_GET['reg_page']) : 1;
$payments_page = isset($_GET['pay_page']) ? max(1, (int)$_GET['pay_page']) : 1;
$items_per_page = 10;

// Get recent registrations
$where_conditions = [];
$params = [];
$types = "";

if ($program_filter > 0) {
    $where_conditions[] = "s.program_id = ?";
    $params[] = $program_filter;
    $types .= "i";
}

if ($year_level_filter > 0) {
    $where_conditions[] = "s.year_level = ?";
    $params[] = $year_level_filter;
    $types .= "i";
}

$where_clause = !empty($where_conditions) ? " WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count for registrations pagination
$count_query = "
    SELECT COUNT(*) as total
    FROM students s 
    JOIN users u ON s.user_id = u.id 
    JOIN programs p ON s.program_id = p.id
    {$where_clause}";

$count_stmt = $conn->prepare($count_query);
if (!empty($params)) {
    $count_stmt->bind_param($types, ...$params);
}
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $items_per_page);

// Get paginated results
$query = "
    SELECT 
        s.id,
        s.student_number,
        s.first_name,
        s.last_name,
        p.program_name,
        s.year_level,
        s.contact_number,
        s.address,
        u.username,
        u.created_at
    FROM students s 
    JOIN users u ON s.user_id = u.id 
    JOIN programs p ON s.program_id = p.id
    {$where_clause}
    ORDER BY {$sort_by} {$sort_direction}
    LIMIT ? OFFSET ?";

$stmt = $conn->prepare($query);
$params[] = $items_per_page;
$params[] = $offset;
$types .= "ii";
$stmt->bind_param($types, ...$params);
$stmt->execute();
$recent_students = $stmt->get_result();

// Pagination settings
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Filter settings
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : '';
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';

// Build query conditions
$conditions = [];
$params = [];
$types = "";

if ($status_filter && $status_filter !== 'all') {
    $conditions[] = "p.status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if ($date_from) {
    $conditions[] = "DATE(p.created_at) >= ?";
    $params[] = $date_from;
    $types .= "s";
}

if ($date_to) {
    $conditions[] = "DATE(p.created_at) <= ?";
    $params[] = $date_to;
    $types .= "s";
}

if ($search) {
    $conditions[] = "(s.student_number LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ? OR p.reference_number LIKE ?)";
    $search_param = "%$search%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
    $types .= "ssss";
}

// Construct WHERE clause
$where_clause = !empty($conditions) ? " WHERE " . implode(" AND ", $conditions) : "";

// Get total count for pagination
$count_query = "
    SELECT COUNT(*) as total
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    $where_clause
";

$stmt = $conn->prepare($count_query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$total_records = $stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $per_page);

// Get paginated results
$query = "
    SELECT 
        p.*,
        s.student_number,
        s.first_name,
        s.last_name,
        ft.name as fee_type,
        CASE 
            WHEN p.status = 'pending' THEN p.created_at
            ELSE COALESCE(p.payment_date, p.created_at)
        END as effective_date
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    $where_clause
    ORDER BY effective_date DESC
    LIMIT ? OFFSET ?
";

// Add pagination parameters
$params[] = $per_page;
$params[] = $offset;
$types .= "ii";

$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$recent_payments = $stmt->get_result();

$page_title = "Admin Dashboard";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-xl-3 col-sm-6 mb-4">
            <div class="card">
                <div class="card-body p-3">
                    <div class="row">
                        <div class="col-8">
                            <div class="numbers">
                                <p class="text-sm mb-0 text-uppercase font-weight-bold">Total Students</p>
                                <h5 class="font-weight-bolder mb-0">
                                    <?php echo number_format($stats['total_students']); ?>
                                </h5>
                            </div>
                        </div>
                        <div class="col-4 text-end">
                            <div class="icon icon-shape bg-gradient-primary shadow text-center border-radius-md">
                                <i class="fas fa-users opacity-10"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-sm-6 mb-4">
            <div class="card">
                <div class="card-body p-3">
                    <div class="row">
                        <div class="col-8">
                            <div class="numbers">
                                <p class="text-sm mb-0 text-uppercase font-weight-bold">Total Payments</p>
                                <h5 class="font-weight-bolder mb-0">
                                    <?php echo number_format($stats['total_payments']); ?>
                                </h5>
                            </div>
                        </div>
                        <div class="col-4 text-end">
                            <div class="icon icon-shape bg-gradient-success shadow text-center border-radius-md">
                                <i class="fas fa-money-bill opacity-10"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-sm-6 mb-4">
            <div class="card">
                <div class="card-body p-3">
                    <div class="row">
                        <div class="col-8">
                            <div class="numbers">
                                <p class="text-sm mb-0 text-uppercase font-weight-bold">Pending Payments</p>
                                <h5 class="font-weight-bolder mb-0">
                                    <?php echo number_format($stats['pending_payments']); ?>
                                </h5>
                            </div>
                        </div>
                        <div class="col-4 text-end">
                            <div class="icon icon-shape bg-gradient-warning shadow text-center border-radius-md">
                                <i class="fas fa-clock opacity-10"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-sm-6 mb-4">
            <div class="card">
                <div class="card-body p-3">
                    <div class="row">
                        <div class="col-8">
                            <div class="numbers">
                                <p class="text-sm mb-0 text-uppercase font-weight-bold">Total Collected</p>
                                <h5 class="font-weight-bolder mb-0">
                                    <?php echo formatMoney($stats['total_collected']); ?>
                                </h5>
                            </div>
                        </div>
                        <div class="col-4 text-end">
                            <div class="icon icon-shape bg-gradient-info shadow text-center border-radius-md">
                                <i class="fas fa-peso-sign opacity-10"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-sm-6 mb-4">
            <div class="card">
                <div class="card-body p-3">
                    <div class="row">
                        <div class="col-8">
                            <div class="numbers">
                                <p class="text-sm mb-0 text-uppercase font-weight-bold">Refunded Amount</p>
                                <h5 class="font-weight-bolder mb-0">
                                    <?php echo formatMoney($stats['total_refunded']); ?>
                                </h5>
                                <small class="text-warning"><?php echo $stats['refunded_payments']; ?> refunds processed</small>
                            </div>
                        </div>
                        <div class="col-4 text-end">
                            <div class="icon icon-shape bg-gradient-warning shadow text-center border-radius-md">
                                <i class="fas fa-undo opacity-10"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="d-flex gap-3">
                        <a href="manage_refunds.php" class="btn btn-warning">
                            <i class="fas fa-undo me-2"></i>View Refunds
                        </a>
                        <a href="manage_students.php" class="btn btn-primary">
                            <i class="fas fa-users me-2"></i>Manage Students
                        </a>
                        <a href="manage_programs.php" class="btn btn-info">
                            <i class="fas fa-graduation-cap me-2"></i>Manage Programs
                        </a>
                        <a href="manage_fee_types.php" class="btn btn-success">
                            <i class="fas fa-money-bill me-2"></i>Manage Fees
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-light">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Recent Registrations</h5>
                        <!-- Removed view all students button -->
                    </div>
                    <div class="mt-3">
                        <form method="GET" class="row g-3">
                            <input type="hidden" name="sort" value="<?php echo htmlspecialchars($sort_column); ?>">
                            <input type="hidden" name="order" value="<?php echo htmlspecialchars($sort_order); ?>">
                            
                            <div class="col-md-4">
                                <select name="program" class="form-select">
                                    <option value="0">All Programs</option>
                                    <?php foreach ($programs as $program): ?>
                                        <option value="<?php echo $program['id']; ?>" 
                                            <?php echo $program_filter == $program['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($program['program_name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-4">
                                <select name="year_level" class="form-select">
                                    <option value="0">All Year Levels</option>
                                    <?php for($i = 1; $i <= 5; $i++): ?>
                                        <option value="<?php echo $i; ?>" 
                                            <?php echo $year_level_filter == $i ? 'selected' : ''; ?>>
                                            Year <?php echo $i; ?>
                                        </option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-4">
                                <button type="submit" class="btn btn-primary">Apply Filters</button>
                                <a href="?sort=created_at&order=DESC" class="btn btn-outline-secondary">Clear Filters</a>
                            </div>
                        </form>
                    </div>
                </div>
                <div class="table-responsive">
                    <table class="table align-items-center mb-0">
                        <thead>
                            <tr>
                                <th>
                                    <a href="?sort=student_number&order=<?php echo ($sort_column === 'student_number' && $sort_order === 'ASC') ? 'DESC' : 'ASC'; ?>" 
                                       class="text-dark text-decoration-none d-flex align-items-center gap-2">
                                        Student
                                        <?php if ($sort_column === 'student_number'): ?>
                                            <i class="fas fa-sort-<?php echo $sort_order === 'ASC' ? 'up' : 'down'; ?>"></i>
                                        <?php else: ?>
                                            <i class="fas fa-sort text-muted"></i>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>
                                    <a href="?sort=program&order=<?php echo ($sort_column === 'program' && $sort_order === 'ASC') ? 'DESC' : 'ASC'; ?>" 
                                       class="text-dark text-decoration-none d-flex align-items-center gap-2">
                                        Course
                                        <?php if ($sort_column === 'program'): ?>
                                            <i class="fas fa-sort-<?php echo $sort_order === 'ASC' ? 'up' : 'down'; ?>"></i>
                                        <?php else: ?>
                                            <i class="fas fa-sort text-muted"></i>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>Year Level</th>
                                <th>
                                    <a href="?sort=created_at&order=<?php echo ($sort_column === 'created_at' && $sort_order === 'ASC') ? 'DESC' : 'ASC'; ?>" 
                                       class="text-dark text-decoration-none d-flex align-items-center gap-2">
                                        Date Registered
                                        <?php if ($sort_column === 'created_at'): ?>
                                            <i class="fas fa-sort-<?php echo $sort_order === 'ASC' ? 'up' : 'down'; ?>"></i>
                                        <?php else: ?>
                                            <i class="fas fa-sort text-muted"></i>
                                        <?php endif; ?>
                                    </a>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($student = $recent_students->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <h6 class="mb-0 text-sm"><?php echo $student['first_name'] . ' ' . $student['last_name']; ?></h6>
                                                <p class="text-xs text-secondary mb-0"><?php echo $student['student_number']; ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <p class="text-sm font-weight-bold mb-0"><?php echo $student['program_name']; ?></p>
                                    </td>
                                    <td>
                                        <p class="text-sm font-weight-bold mb-0"><?php echo $student['year_level']; ?></p>
                                    </td>
                                    <td>
                                        <p class="text-sm font-weight-bold mb-0"><?php echo date('M d, Y', strtotime($student['created_at'])); ?></p>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($total_pages > 1): ?>
                <div class="card-footer">
                    <nav aria-label="Page navigation">
                        <ul class="pagination justify-content-center mb-0">
                            <?php 
                            // Previous page link
                            if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?reg_page=<?php echo ($registrations_page - 1); ?>&sort=<?php echo urlencode($sort_column); ?>&order=<?php echo urlencode($sort_order); ?>&program=<?php echo urlencode($program_filter); ?>&year_level=<?php echo urlencode($year_level_filter); ?>&pay_page=<?php echo $payments_page; ?>" aria-label="Previous">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>

                            <?php
                            // Calculate range of page numbers to show
                            $range = 2;
                            $start_page = max(1, $page - $range);
                            $end_page = min($total_pages, $page + $range);

                            // Show first page if not in range
                            if ($start_page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=1&sort=<?php echo urlencode($sort_column); ?>&order=<?php echo urlencode($sort_order); ?>&program=<?php echo urlencode($program_filter); ?>&year_level=<?php echo urlencode($year_level_filter); ?>">1</a>
                                </li>
                                <?php if ($start_page > 2): ?>
                                    <li class="page-item disabled"><span class="page-link">...</span></li>
                                <?php endif;
                            endif;

                            // Show page numbers
                            for ($i = $start_page; $i <= $end_page; $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?>&sort=<?php echo urlencode($sort_column); ?>&order=<?php echo urlencode($sort_order); ?>&program=<?php echo urlencode($program_filter); ?>&year_level=<?php echo urlencode($year_level_filter); ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor;

                            // Show last page if not in range
                            if ($end_page < $total_pages): 
                                if ($end_page < $total_pages - 1): ?>
                                    <li class="page-item disabled"><span class="page-link">...</span></li>
                                <?php endif; ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $total_pages; ?>&sort=<?php echo urlencode($sort_column); ?>&order=<?php echo urlencode($sort_order); ?>&program=<?php echo urlencode($program_filter); ?>&year_level=<?php echo urlencode($year_level_filter); ?>"><?php echo $total_pages; ?></a>
                                </li>
                            <?php endif;

                            // Next page link
                            if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo ($page + 1); ?>&sort=<?php echo urlencode($sort_column); ?>&order=<?php echo urlencode($sort_order); ?>&program=<?php echo urlencode($program_filter); ?>&year_level=<?php echo urlencode($year_level_filter); ?>" aria-label="Next">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <div class="col-md-6">
            <div class="card">
                <div class="card-header bg-light">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Recent Payments</h5>
                        <a href="manage_payments.php" class="btn btn-primary btn-sm">View All</a>
                    </div>
                </div>
                <div class="table-responsive">
                    <table class="table align-items-center mb-0">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            // Get total count for payments pagination
                            $payments_count_query = "SELECT COUNT(*) as total FROM payments";
                            $payments_total_result = $conn->query($payments_count_query);
                            $payments_total = $payments_total_result->fetch_assoc()['total'];
                            $payments_total_pages = ceil($payments_total / $items_per_page);
                            
                            // Calculate offset for payments
                            $payments_offset = ($payments_page - 1) * $items_per_page;
                            
                            // Get paginated payments
                            $payments_query = "
                                SELECT 
                                    p.*,
                                    s.student_number,
                                    s.first_name,
                                    s.last_name,
                                    ft.name as fee_type,
                                    CASE 
                                        WHEN p.status = 'pending' THEN p.created_at
                                        ELSE COALESCE(p.payment_date, p.created_at)
                                    END as effective_date
                                FROM payments p
                                JOIN student_fees sf ON p.student_fee_id = sf.id
                                JOIN students s ON sf.student_id = s.id
                                JOIN fee_types ft ON sf.fee_type_id = ft.id
                                ORDER BY effective_date DESC
                                LIMIT ? OFFSET ?";
                            
                            $stmt = $conn->prepare($payments_query);
                            $stmt->bind_param("ii", $items_per_page, $payments_offset);
                            $stmt->execute();
                            $recent_payments = $stmt->get_result();
                            
                            while ($payment = $recent_payments->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <h6 class="mb-0 text-sm"><?php echo $payment['first_name'] . ' ' . $payment['last_name']; ?></h6>
                                                <p class="text-xs text-secondary mb-0"><?php echo $payment['student_number']; ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <p class="text-sm font-weight-bold mb-0"><?php echo formatMoney($payment['amount']); ?></p>
                                        <p class="text-xs text-secondary mb-0"><?php echo $payment['fee_type']; ?></p>
                                    </td>
                                    <td>
                                        <span class="badge badge-sm bg-<?php 
                                            echo $payment['status'] === 'confirmed' ? 'success' : 
                                                ($payment['status'] === 'pending' ? 'warning' : 'danger'); 
                                        ?>">
                                            <?php echo ucfirst($payment['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <p class="text-sm font-weight-bold mb-0">
                                            <?php echo formatDateTime($payment['effective_date']); ?>
                                        </p>
                                        <p class="text-xs text-secondary mb-0">
                                            <?php 
                                            if ($payment['status'] === 'pending') {
                                                echo 'Created: ' . formatDateTime($payment['created_at']);
                                            } else {
                                                echo 'Paid: ' . formatDateTime($payment['payment_date']);
                                            }
                                            ?>
                                        </p>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($payments_total_pages > 1): ?>
                <div class="card-footer">
                    <nav aria-label="Payments navigation">
                        <ul class="pagination justify-content-center mb-0">
                            <?php 
                            // Previous page link
                            if ($payments_page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?pay_page=<?php echo ($payments_page - 1); ?>&reg_page=<?php echo $registrations_page; ?>" aria-label="Previous">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>

                            <?php
                            // Calculate range of page numbers to show
                            $range = 2;
                            $start_page = max(1, $payments_page - $range);
                            $end_page = min($payments_total_pages, $payments_page + $range);

                            // Show first page if not in range
                            if ($start_page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?pay_page=1&reg_page=<?php echo $registrations_page; ?>">1</a>
                                </li>
                                <?php if ($start_page > 2): ?>
                                    <li class="page-item disabled"><span class="page-link">...</span></li>
                                <?php endif;
                            endif;

                            // Show page numbers
                            for ($i = $start_page; $i <= $end_page; $i++): ?>
                                <li class="page-item <?php echo $i === $payments_page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?pay_page=<?php echo $i; ?>&reg_page=<?php echo $registrations_page; ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor;

                            // Show last page if not in range
                            if ($end_page < $payments_total_pages): 
                                if ($end_page < $payments_total_pages - 1): ?>
                                    <li class="page-item disabled"><span class="page-link">...</span></li>
                                <?php endif; ?>
                                <li class="page-item">
                                    <a class="page-link" href="?pay_page=<?php echo $payments_total_pages; ?>&reg_page=<?php echo $registrations_page; ?>"><?php echo $payments_total_pages; ?></a>
                                </li>
                            <?php endif;

                            // Next page link
                            if ($payments_page < $payments_total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?pay_page=<?php echo ($payments_page + 1); ?>&reg_page=<?php echo $registrations_page; ?>" aria-label="Next">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>