<?php
// Initialize required files
require_once "../config/config.php";
require_once "../controllers/AuthController.php";
require_once "../includes/functions.php";
require_once "../includes/Database.php";

session_start();

// Basic access control
if (!isset($_SESSION["user_id"]) || $_SESSION["role"] != "admin") {
    header("Location: ../login.php");
    exit();
}

$db = Database::getInstance();
$conn = $db->getConnection();

// Initialize messages
$success_message = isset($_SESSION["success_message"]) ? $_SESSION["success_message"] : "";
$error_message = isset($_SESSION["error_message"]) ? $_SESSION["error_message"] : "";
unset($_SESSION["success_message"], $_SESSION["error_message"]);

// Get student data
$student_id = isset($_GET["id"]) ? intval($_GET["id"]) : 0;
$student = null;

if ($student_id) {
    $stmt = $conn->prepare("SELECT s.*, u.email, u.status FROM students s 
                           JOIN users u ON s.user_id = u.id 
                           WHERE s.id = ?");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
}

if (!$student) {
    $_SESSION["error_message"] = "Student not found.";
    header("Location: manage_students.php");
    exit();
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    try {
        $db->beginTransaction();

        // Validate input
        if (empty($_POST["student_number"]) || empty($_POST["first_name"]) || 
            empty($_POST["last_name"]) || empty($_POST["email"]) || 
            !isset($_POST["program_id"]) || !isset($_POST["year_level"])) {
            throw new Exception("All fields are required.");
        }

        $student_number = trim($_POST["student_number"]);
        $first_name = trim($_POST["first_name"]);
        $last_name = trim($_POST["last_name"]);
        $email = trim($_POST["email"]);
        $program_id = intval($_POST["program_id"]);
        $year_level = intval($_POST["year_level"]);
        $status = $_POST["status"] === "active" ? "active" : "inactive";

        // Check if student number is unique (excluding current student)
        $stmt = $conn->prepare("SELECT id FROM students WHERE student_number = ? AND id != ?");
        $stmt->bind_param("si", $student_number, $student_id);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            throw new Exception("Student number already exists.");
        }

        // Check if email is unique (excluding current student)
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
        $stmt->bind_param("si", $email, $student["user_id"]);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            throw new Exception("Email address already exists.");
        }

        // Update student data
        $stmt = $conn->prepare("UPDATE students SET 
            student_number = ?, 
            first_name = ?, 
            last_name = ?, 
            program_id = ?, 
            year_level = ? 
            WHERE id = ?");
        $stmt->bind_param("sssiii", 
            $student_number, 
            $first_name, 
            $last_name, 
            $program_id, 
            $year_level, 
            $student_id
        );
        $stmt->execute();

        // Update user data
        $stmt = $conn->prepare("UPDATE users SET email = ?, status = ? WHERE id = ?");
        $stmt->bind_param("ssi", $email, $status, $student["user_id"]);
        $stmt->execute();

        $db->commit();
        $_SESSION["success_message"] = "Student updated successfully.";
        header("Location: manage_students.php");
        exit();

    } catch (Exception $e) {
        $db->rollback();
        $error_message = $e->getMessage();
    }
}

$page_title = "Edit Student";
require_once "../includes/header.php";

// Get programs for dropdown
$programs = $conn->query("SELECT * FROM programs ORDER BY program_code");
?>

<div class="container-fluid py-4">
    <div class="card">
        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Edit Student</h5>
            <a href="manage_students.php" class="btn btn-light">
                <i class="fas fa-arrow-left"></i> Back to List
            </a>
        </div>

        <div class="card-body">
            <?php if ($error_message): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>

            <form method="POST" class="row g-3">
                <div class="col-md-6">
                    <label for="student_number" class="form-label">Student Number</label>
                    <input type="text" class="form-control" id="student_number" name="student_number"
                           value="<?php echo htmlspecialchars($student["student_number"]); ?>" required>
                </div>

                <div class="col-md-6">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" class="form-control" id="email" name="email"
                           value="<?php echo htmlspecialchars($student["email"]); ?>" required>
                </div>

                <div class="col-md-6">
                    <label for="first_name" class="form-label">First Name</label>
                    <input type="text" class="form-control" id="first_name" name="first_name"
                           value="<?php echo htmlspecialchars($student["first_name"]); ?>" required>
                </div>

                <div class="col-md-6">
                    <label for="last_name" class="form-label">Last Name</label>
                    <input type="text" class="form-control" id="last_name" name="last_name"
                           value="<?php echo htmlspecialchars($student["last_name"]); ?>" required>
                </div>

                <div class="col-md-4">
                    <label for="program_id" class="form-label">Program</label>
                    <select class="form-select" id="program_id" name="program_id" required>
                        <option value="">Select Program</option>
                        <?php while ($program = $programs->fetch_assoc()): ?>
                            <option value="<?php echo $program["id"]; ?>"
                                <?php echo ($student["program_id"] == $program["id"]) ? "selected" : ""; ?>>
                                <?php echo htmlspecialchars($program["program_code"]); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>

                <div class="col-md-4">
                    <label for="year_level" class="form-label">Year Level</label>
                    <select class="form-select" id="year_level" name="year_level" required>
                        <option value="">Select Level</option>
                        <?php for ($i = 1; $i <= 4; $i++): ?>
                            <option value="<?php echo $i; ?>"
                                <?php echo ($student["year_level"] == $i) ? "selected" : ""; ?>>
                                Level <?php echo $i; ?>
                            </option>
                        <?php endfor; ?>
                    </select>
                </div>

                <div class="col-md-4">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-select" id="status" name="status" required>
                        <option value="active" <?php echo ($student["status"] === "active") ? "selected" : ""; ?>>Active</option>
                        <option value="inactive" <?php echo ($student["status"] === "inactive") ? "selected" : ""; ?>>Inactive</option>
                    </select>
                </div>

                <div class="col-12">
                    <hr>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Update Student
                    </button>
                    <a href="manage_students.php" class="btn btn-secondary">
                        <i class="fas fa-times"></i> Cancel
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once "../includes/footer.php"; ?>