<?php
require_once '../config/config.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    exit('Unauthorized');
}

$program_id = intval($_GET['program_id'] ?? 0);
$year_level = intval($_GET['year_level'] ?? 0);

if (!$program_id || !$year_level) {
    exit('Invalid parameters');
}

// Get students eligible for promotion
$stmt = $conn->prepare("
    SELECT s.*, u.username, u.email, p.program_code, p.program_name,
           COALESCE(SUM(CASE WHEN sf.status != 'paid' THEN 1 ELSE 0 END), 0) as pending_fees
    FROM students s
    JOIN users u ON s.user_id = u.id
    JOIN programs p ON s.program_id = p.id
    LEFT JOIN student_fees sf ON s.id = sf.student_id 
        AND sf.academic_year_id = s.academic_year_id
    WHERE s.program_id = ? 
    AND s.year_level = ?
    GROUP BY s.id, s.student_number, s.first_name, s.last_name, s.year_level, 
             s.program_id, u.username, u.email, p.program_code, p.program_name
    ORDER BY s.last_name, s.first_name
");

$stmt->bind_param("ii", $program_id, $year_level);
$stmt->execute();
$students = $stmt->get_result();

if ($students->num_rows === 0) {
    echo '<div class="alert alert-info">No students found in this course and year level.</div>';
    exit;
}
?>

<table class="table align-items-center mb-0">
    <thead>
        <tr>
            <th>Student Number</th>
            <th>Name</th>
            <th>Current Year</th>
            <th>New Year</th>
            <th>Status</th>
        </tr>
    </thead>
    <tbody>
        <?php while ($student = $students->fetch_assoc()): ?>
            <tr>
                <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></td>
                <td><?php echo $student['year_level']; ?></td>
                <td><?php echo $student['year_level'] < 5 ? $student['year_level'] + 1 : 'Max Level'; ?></td>
                <td>
                    <?php if ($student['year_level'] >= 5): ?>
                        <span class="badge bg-warning">Maximum Year Level</span>
                    <?php else: ?>
                        <span class="badge bg-success">Eligible for Promotion</span>
                        <?php if ($student['pending_fees'] > 0): ?>
                            <br><small class="text-warning">(Has Pending Fees)</small>
                        <?php endif; ?>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endwhile; ?>
    </tbody>
</table>