<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Initialize message
$message = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $conn->begin_transaction();

        $action = $_POST['action'] ?? '';

        switch ($action) {
            case 'add':
                $year_name = $_POST['year_name'];
                $start_date = $_POST['start_date'];
                $end_date = $_POST['end_date'];
                $is_current = isset($_POST['is_current']) ? 1 : 0;

                if (!$year_name || !$start_date || !$end_date) {
                    throw new Exception("All fields are required.");
                }

                // Validate dates
                if (strtotime($end_date) <= strtotime($start_date)) {
                    throw new Exception("End date must be after start date.");
                }

                // If setting as current year, unset any existing current year
                if ($is_current) {
                    $conn->query("UPDATE academic_years SET is_current = 0");
                }

                // Insert new academic year
                $stmt = $conn->prepare("
                    INSERT INTO academic_years (year_name, start_date, end_date, is_current) 
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->bind_param("sssi", $year_name, $start_date, $end_date, $is_current);
                
                if (!$stmt->execute()) {
                    throw new Exception("Error adding academic year: " . $stmt->error);
                }

                $message = '<div class="alert alert-success">Academic year added successfully!</div>';
                break;

            case 'edit':
                $year_id = filter_input(INPUT_POST, 'year_id', FILTER_VALIDATE_INT);
                $year_name = filter_input(INPUT_POST, 'year_name', FILTER_SANITIZE_STRING);
                $start_date = filter_input(INPUT_POST, 'start_date', FILTER_SANITIZE_STRING);
                $end_date = filter_input(INPUT_POST, 'end_date', FILTER_SANITIZE_STRING);
                $is_current = isset($_POST['is_current']) ? 1 : 0;
                $status = filter_input(INPUT_POST, 'status', FILTER_SANITIZE_STRING);

                if (!$year_id || !$year_name || !$start_date || !$end_date || !$status) {
                    throw new Exception("All fields are required.");
                }

                // Validate dates
                if (strtotime($end_date) <= strtotime($start_date)) {
                    throw new Exception("End date must be after start date.");
                }

                // If setting as current year, unset any existing current year
                if ($is_current) {
                    $stmt = $conn->prepare("UPDATE academic_years SET is_current = 0 WHERE id != ?");
                    $stmt->bind_param("i", $year_id);
                    $stmt->execute();
                }

                // Update academic year
                $stmt = $conn->prepare("
                    UPDATE academic_years 
                    SET year_name = ?, start_date = ?, end_date = ?, 
                        is_current = ?, status = ? 
                    WHERE id = ?
                ");
                $stmt->bind_param("sssisi", 
                    $year_name, $start_date, $end_date, 
                    $is_current, $status, $year_id
                );
                
                if (!$stmt->execute()) {
                    throw new Exception("Error updating academic year: " . $stmt->error);
                }

                $_SESSION['success_message'] = "Academic year updated successfully!";
                break;

            case 'delete':
                $year_id = filter_input(INPUT_POST, 'year_id', FILTER_VALIDATE_INT);

                if (!$year_id) {
                    throw new Exception("Invalid academic year ID.");
                }

                // Check if year has associated programs
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as program_count 
                    FROM programs 
                    WHERE academic_year_id = ?
                ");
                $stmt->bind_param("i", $year_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $program_count = $result->fetch_assoc()['program_count'];

                if ($program_count > 0) {
                    throw new Exception("Cannot delete academic year: {$program_count} program(s) are associated with it.");
                }

                // Delete academic year
                $stmt = $conn->prepare("DELETE FROM academic_years WHERE id = ?");
                $stmt->bind_param("i", $year_id);
                
                if (!$stmt->execute()) {
                    throw new Exception("Error deleting academic year: " . $stmt->error);
                }

                $message = '<div class="alert alert-success">Academic year deleted successfully!</div>';
                break;

            default:
                throw new Exception("Invalid action.");
        }

        $conn->commit();

    } catch (Exception $e) {
        $conn->rollback();
        $message = '<div class="alert alert-danger">Error: ' . $e->getMessage() . '</div>';
    }

    // Redirect to prevent form resubmission
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Get all academic years
$academic_years = $conn->query("
    SELECT * FROM academic_years 
    ORDER BY start_date DESC
");

$page_title = "Manage Academic Years";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <?php if ($message): ?>
        <?php echo $message; ?>
    <?php endif; ?>

    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0 d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">Academic Years</h6>
                    <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addYearModal">
                        Add Academic Year
                    </button>
                </div>
                <div class="card-body px-0 pt-0 pb-2">
                    <div class="table-responsive p-0">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th>Year Name</th>
                                    <th>Start Date</th>
                                    <th>End Date</th>
                                    <th>Status</th>
                                    <th>Current</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($year = $academic_years->fetch_assoc()): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($year['year_name']); ?></td>
                                        <td><?php echo date('M d, Y', strtotime($year['start_date'])); ?></td>
                                        <td><?php echo date('M d, Y', strtotime($year['end_date'])); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $year['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                <?php echo ucfirst($year['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($year['is_current']): ?>
                                                <span class="badge bg-info">Current</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button type="button" class="btn btn-info btn-sm" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#editYearModal" 
                                                    data-year='<?php echo json_encode($year); ?>'>
                                                Edit
                                            </button>
                                            <button type="button" class="btn btn-danger btn-sm"
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#deleteYearModal"
                                                    data-year-id="<?php echo $year['id']; ?>"
                                                    data-year-name="<?php echo htmlspecialchars($year['year_name']); ?>">
                                                Delete
                                            </button>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Academic Year Modal -->
<div class="modal fade" id="addYearModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Academic Year</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <input type="hidden" name="action" value="add">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="year_name" class="form-label">Year Name</label>
                        <input type="text" class="form-control" id="year_name" name="year_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date" required>
                    </div>
                    <div class="mb-3">
                        <label for="end_date" class="form-label">End Date</label>
                        <input type="date" class="form-control" id="end_date" name="end_date" required>
                    </div>
                    <div class="mb-3 form-check">
                        <input type="checkbox" class="form-check-input" id="is_current" name="is_current">
                        <label class="form-check-label" for="is_current">Set as Current Academic Year</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Add Academic Year</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Academic Year Modal -->
<div class="modal fade" id="editYearModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Academic Year</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="year_id" id="edit_year_id">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_year_name" class="form-label">Year Name</label>
                        <input type="text" class="form-control" id="edit_year_name" name="year_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_start_date" class="form-label">Start Date</label>
                        <input type="date" class="form-control" id="edit_start_date" name="start_date" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_end_date" class="form-label">End Date</label>
                        <input type="date" class="form-control" id="edit_end_date" name="end_date" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_status" class="form-label">Status</label>
                        <select class="form-select" id="edit_status" name="status" required>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                    <div class="mb-3 form-check">
                        <input type="checkbox" class="form-check-input" id="edit_is_current" name="is_current">
                        <label class="form-check-label" for="edit_is_current">Set as Current Academic Year</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Update Academic Year</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Academic Year Modal -->
<div class="modal fade" id="deleteYearModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Delete Academic Year</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="year_id" id="delete_year_id">
                <div class="modal-body">
                    <p>Are you sure you want to delete academic year <strong id="delete_year_name"></strong>?</p>
                    <p class="text-danger">This action cannot be undone!</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Academic Year</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Handle Edit Modal
    const editYearModal = document.getElementById('editYearModal');
    editYearModal.addEventListener('show.bs.modal', function(event) {
        const button = event.relatedTarget;
        const yearData = JSON.parse(button.getAttribute('data-year'));
        
        document.getElementById('edit_year_id').value = yearData.id;
        document.getElementById('edit_year_name').value = yearData.year_name;
        document.getElementById('edit_start_date').value = yearData.start_date;
        document.getElementById('edit_end_date').value = yearData.end_date;
        document.getElementById('edit_status').value = yearData.status;
        document.getElementById('edit_is_current').checked = yearData.is_current == 1;
    });

    // Handle Delete Modal
    const deleteYearModal = document.getElementById('deleteYearModal');
    deleteYearModal.addEventListener('show.bs.modal', function(event) {
        const button = event.relatedTarget;
        const yearId = button.getAttribute('data-year-id');
        const yearName = button.getAttribute('data-year-name');
        
        document.getElementById('delete_year_id').value = yearId;
        document.getElementById('delete_year_name').textContent = yearName;
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>