<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $name = $conn->real_escape_string($_POST['name']);
                $amount = floatval($_POST['amount']);
                $level = $conn->real_escape_string($_POST['level']);
                
                $sql = "INSERT INTO fee_types (name, amount, level) VALUES (?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("sds", $name, $amount, $level);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Fee type added successfully!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error adding fee type.</div>';
                }
                break;
                
            case 'edit':
                $id = intval($_POST['id']);
                $name = $conn->real_escape_string($_POST['name']);
                $amount = floatval($_POST['amount']);
                $level = $conn->real_escape_string($_POST['level']);
                
                $sql = "UPDATE fee_types SET name = ?, amount = ?, level = ? WHERE id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("sdsi", $name, $amount, $level, $id);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Fee type updated successfully!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error updating fee type.</div>';
                }
                break;
                
            case 'delete':
                $id = intval($_POST['id']);
                
                // Check if fee type is in use
                $check = $conn->query("SELECT COUNT(*) as count FROM student_fees WHERE fee_type_id = $id");
                $inUse = $check->fetch_assoc()['count'] > 0;
                
                if ($inUse) {
                    $message = '<div class="alert alert-danger">Cannot delete fee type as it is currently in use.</div>';
                } else {
                    if ($conn->query("DELETE FROM fee_types WHERE id = $id")) {
                        $message = '<div class="alert alert-success">Fee type deleted successfully!</div>';
                    } else {
                        $message = '<div class="alert alert-danger">Error deleting fee type.</div>';
                    }
                }
                break;
        }
    }
}

// Get all fee types with their program assignments
$query = "
    SELECT 
        ft.*, 
        GROUP_CONCAT(DISTINCT p.program_code) as assigned_programs,
        GROUP_CONCAT(DISTINCT ay.year_name) as assigned_years
    FROM fee_types ft
    LEFT JOIN student_fees sf ON ft.id = sf.fee_type_id
    LEFT JOIN students s ON sf.student_id = s.id
    LEFT JOIN programs p ON s.program_id = p.id
    LEFT JOIN academic_years ay ON sf.academic_year_id = ay.id
    GROUP BY ft.id
    ORDER BY ft.name
";
$fee_types = $conn->query($query);

$page_title = "Manage Fee Types";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Manage Fee Types</h6>
                        <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addFeeTypeModal">
                            Add New Fee Type
                        </button>
                    </div>
                </div>
                <div class="card-body px-0 pt-0 pb-2">
                    <?php echo $message; ?>
                    <div class="table-responsive p-0">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Amount</th>
                                    <th>Level</th>
                                    <th>Assigned Programs</th>
                                    <th>Academic Years</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($fee_type = $fee_types->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($fee_type['name']); ?></td>
                                    <td><?php echo formatMoney($fee_type['amount']); ?></td>
                                    <td><?php echo htmlspecialchars($fee_type['level']); ?></td>
                                    <td><?php echo htmlspecialchars($fee_type['assigned_programs'] ?: 'None'); ?></td>
                                    <td><?php echo htmlspecialchars($fee_type['assigned_years'] ?: 'None'); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-info btn-sm" 
                                                onclick="editFeeType(<?php echo htmlspecialchars(json_encode($fee_type)); ?>)">
                                            Edit
                                        </button>
                                        <button type="button" class="btn btn-danger btn-sm" 
                                                onclick="confirmDelete(<?php echo $fee_type['id']; ?>, '<?php echo htmlspecialchars($fee_type['name']); ?>')">
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Fee Type Modal -->
<div class="modal fade" id="addFeeTypeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Fee Type</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    <div class="mb-3">
                        <label for="name" class="form-label">Name</label>
                        <input type="text" class="form-control" id="name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="amount" class="form-label">Amount</label>
                        <input type="number" class="form-control" id="amount" name="amount" step="0.01" required>
                    </div>
                    <div class="mb-3">
                        <label for="level" class="form-label">Level</label>
                        <textarea class="form-control" id="level" name="level" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Add Fee Type</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Fee Type Modal -->
<div class="modal fade" id="editFeeTypeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Fee Type</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit_id">
                    <div class="mb-3">
                        <label for="edit_name" class="form-label">Name</label>
                        <input type="text" class="form-control" id="edit_name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_amount" class="form-label">Amount</label>
                        <input type="number" class="form-control" id="edit_amount" name="amount" step="0.01" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_level" class="form-label">Level</label>
                        <textarea class="form-control" id="edit_level" name="level" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteFeeTypeModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete the fee type "<span id="delete_fee_type_name"></span>"?
            </div>
            <div class="modal-footer">
                <form action="" method="POST">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="delete_id">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
function editFeeType(feeType) {
    document.getElementById('edit_id').value = feeType.id;
    document.getElementById('edit_name').value = feeType.name;
    document.getElementById('edit_amount').value = feeType.amount;
    document.getElementById('edit_level').value = feeType.level;
    new bootstrap.Modal(document.getElementById('editFeeTypeModal')).show();
}

function confirmDelete(id, name) {
    document.getElementById('delete_id').value = id;
    document.getElementById('delete_fee_type_name').textContent = name;
    new bootstrap.Modal(document.getElementById('deleteFeeTypeModal')).show();
}
</script>

<?php require_once '../includes/footer.php'; ?>