<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';
require_once '../includes/program_fees.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$success_message = '';
$error_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'assign':
                try {
                    $program_id = $_POST['program_id'];
                    $fee_types = isset($_POST['fee_types']) ? $_POST['fee_types'] : [];
                    $academic_year_id = $_POST['academic_year_id'];
                    $apply_to_existing = isset($_POST['apply_to_existing']) ? true : false;

                    $conn->begin_transaction();

                    // Remove existing assignments for this program and academic year
                    $stmt = $conn->prepare("
                        DELETE FROM program_fees 
                        WHERE program_id = ? AND academic_year_id = ?
                    ");
                    $stmt->bind_param("ii", $program_id, $academic_year_id);
                    $stmt->execute();

                    // Insert new assignments
                    if (!empty($fee_types)) {
                        $stmt = $conn->prepare("
                            INSERT INTO program_fees (program_id, fee_type_id, academic_year_id)
                            VALUES (?, ?, ?)
                        ");
                        foreach ($fee_types as $fee_type_id) {
                            $stmt->bind_param("iii", $program_id, $fee_type_id, $academic_year_id);
                            $stmt->execute();
                        }

                        // If apply to existing students is checked
                        if ($apply_to_existing) {
                            require_once '../includes/fee_sync.php';
                            
                            // Get all students in the program
                            $stmt = $conn->prepare("
                                SELECT id 
                                FROM students 
                                WHERE program_id = ?
                            ");
                            $stmt->bind_param("i", $program_id);
                            $stmt->execute();
                            $students = $stmt->get_result();

                            while ($student = $students->fetch_assoc()) {
                                syncStudentFees($conn, $student['id']);
                            }
                        }
                    }

                    $conn->commit();
                    $success_message = "Program fees assigned successfully.";
                } catch (Exception $e) {
                    $conn->rollback();
                    $error_message = "Error: " . $e->getMessage();
                }
                break;

            case 'bulk_assign':
                try {
                    $academic_year_id = $_POST['bulk_academic_year_id'];
                    $program_ids = $_POST['bulk_programs'] ?? [];
                    $fee_type_ids = $_POST['bulk_fee_types'] ?? [];

                    if (empty($program_ids) || empty($fee_type_ids)) {
                        throw new Exception("Please select both programs and fees");
                    }

                    $conn->begin_transaction();

                    foreach ($program_ids as $program_id) {
                        // Remove existing assignments
                        $stmt = $conn->prepare("
                            DELETE FROM program_fees 
                            WHERE program_id = ? AND academic_year_id = ?
                        ");
                        $stmt->bind_param("ii", $program_id, $academic_year_id);
                        $stmt->execute();

                        // Insert new assignments
                        $stmt = $conn->prepare("
                            INSERT INTO program_fees (program_id, fee_type_id, academic_year_id)
                            VALUES (?, ?, ?)
                        ");
                        foreach ($fee_type_ids as $fee_type_id) {
                            $stmt->bind_param("iii", $program_id, $fee_type_id, $academic_year_id);
                            $stmt->execute();
                        }
                    }

                    $conn->commit();
                    $success_message = "Bulk fee assignment completed successfully.";
                } catch (Exception $e) {
                    $conn->rollback();
                    $error_message = "Error: " . $e->getMessage();
                }
                break;

            case 'delete_assignment':
                try {
                    $program_id = intval($_POST['program_id'] ?? 0);
                    $academic_year_id = intval($_POST['academic_year_id'] ?? 0);
                    
                    if (!$program_id || !$academic_year_id) {
                        throw new Exception("Invalid program or academic year ID.");
                    }

                    // Check if any students have paid fees in this program and academic year
                    $check_stmt = $conn->prepare("
                        SELECT COUNT(*) as count
                        FROM student_fees sf
                        JOIN students s ON sf.student_id = s.id
                        WHERE s.program_id = ? 
                        AND sf.academic_year_id = ?
                        AND sf.payment_status = 'paid'
                    ");
                    $check_stmt->bind_param("ii", $program_id, $academic_year_id);
                    $check_stmt->execute();
                    $paid_count = $check_stmt->get_result()->fetch_assoc()['count'];

                    if ($paid_count > 0) {
                        throw new Exception("Cannot delete: Some students have already paid fees for this program in the selected academic year.");
                    }

                    // Delete the fee assignments
                    $stmt = $conn->prepare("
                        DELETE FROM program_fees 
                        WHERE program_id = ? AND academic_year_id = ?
                    ");
                    $stmt->bind_param("ii", $program_id, $academic_year_id);
                    
                    if ($stmt->execute()) {
                        $success_message = "Fee assignments deleted successfully.";
                    } else {
                        throw new Exception("Error deleting fee assignments: " . $stmt->error);
                    }
                } catch (Exception $e) {
                    $error_message = "Error: " . $e->getMessage();
                }
                break;
        }
    }
}

// Get filter values
$filter_program = isset($_GET['filter_program']) ? intval($_GET['filter_program']) : 0;
$filter_academic_year = isset($_GET['filter_academic_year']) ? intval($_GET['filter_academic_year']) : 0;
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Get all active programs
$programs = $conn->query("
    SELECT id, program_code, program_name 
    FROM programs 
    WHERE status = 'active'
    ORDER BY program_name
");

// Get all active fee types
$fee_types = $conn->query("
    SELECT id, name, amount, level 
    FROM fee_types 
    WHERE status = 'active'
    ORDER BY name
");

// Get all active academic years
$academic_years = $conn->query("
    SELECT id, year_name 
    FROM academic_years 
    WHERE status = 'active'
    ORDER BY year_name DESC
");

// Get current program fee assignments
$program_fees = [];
$result = $conn->query("
    SELECT 
        pf.program_id,
        pf.academic_year_id,
        GROUP_CONCAT(pf.fee_type_id) as fee_type_ids
    FROM program_fees pf
    GROUP BY pf.program_id, pf.academic_year_id
");
while ($row = $result->fetch_assoc()) {
    $key = $row['program_id'] . '_' . $row['academic_year_id'];
    $program_fees[$key] = explode(',', $row['fee_type_ids']);
}

$page_title = "Admin - Manage Program Fees";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <style>
        .fees-list {
            max-height: 200px;
            overflow-y: auto;
            padding-right: 10px;
        }
        .fees-list::-webkit-scrollbar {
            width: 5px;
        }
        .fees-list::-webkit-scrollbar-thumb {
            background: #888;
            border-radius: 5px;
        }
        .table td {
            vertical-align: middle;
        }
    </style>

    <?php if ($success_message): ?>
        <div class="alert alert-success"><?php echo $success_message; ?></div>
    <?php endif; ?>
    <?php if ($error_message): ?>
        <div class="alert alert-danger"><?php echo $error_message; ?></div>
    <?php endif; ?>

    <!-- Bulk Assignment Section -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">Bulk Fee Assignment</h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="bulk_assign">
                        
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label class="form-label">Academic Year</label>
                                <select class="form-select" name="bulk_academic_year_id" required>
                                    <option value="">Select Academic Year</option>
                                    <?php 
                                    $academic_years->data_seek(0);
                                    while ($year = $academic_years->fetch_assoc()): 
                                    ?>
                                    <option value="<?php echo $year['id']; ?>">
                                        <?php echo htmlspecialchars($year['year_name']); ?>
                                    </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label">Select Programs</label>
                                <div class="card">
                                    <div class="card-body" style="max-height: 300px; overflow-y: auto;">
                                        <?php 
                                        $programs->data_seek(0);
                                        while ($program = $programs->fetch_assoc()): 
                                        ?>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" 
                                                   name="bulk_programs[]" 
                                                   value="<?php echo $program['id']; ?>"
                                                   id="prog_<?php echo $program['id']; ?>">
                                            <label class="form-check-label" for="prog_<?php echo $program['id']; ?>">
                                                <?php echo htmlspecialchars($program['program_name']); ?>
                                                (<?php echo htmlspecialchars($program['program_code']); ?>)
                                            </label>
                                        </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <label class="form-label">Select Fees</label>
                                <div class="card">
                                    <div class="card-body" style="max-height: 300px; overflow-y: auto;">
                                        <style>
                                            .form-check {
                                                padding: 10px;
                                                border-bottom: 1px solid #eee;
                                                margin-bottom: 5px;
                                            }
                                            .form-check:last-child {
                                                border-bottom: none;
                                            }
                                            .form-check:hover {
                                                background-color: #f8f9fa;
                                            }
                                        </style>
                                        <?php 
                                        $fee_types->data_seek(0);
                                        while ($fee = $fee_types->fetch_assoc()): 
                                        ?>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" 
                                                   name="bulk_fee_types[]" 
                                                   value="<?php echo $fee['id']; ?>"
                                                   id="fee_<?php echo $fee['id']; ?>">
                                            <label class="form-check-label" for="fee_<?php echo $fee['id']; ?>">
                                                <strong><?php echo htmlspecialchars($fee['name']); ?></strong>
                                                <br>
                                                <small class="text-muted">
                                                    <?php echo htmlspecialchars($fee['level'] ?: 'No level available'); ?>
                                                </small>
                                                <br>
                                                <span class="badge bg-primary">GHs <?php echo number_format($fee['amount'], 2); ?></span>
                                            </label>
                                        </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <button type="submit" class="btn btn-primary">
                                    Bulk Assign Fees
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Current Fee Assignments -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">Current Fee Assignments</h5>
                    <div class="d-flex gap-2">
                        <form method="GET" class="row g-2 align-items-center">
                            <div class="col-auto">
                                <select name="filter_program" class="form-select">
                                    <option value="">All Programs</option>
                                    <?php 
                                    $programs->data_seek(0);
                                    while ($program = $programs->fetch_assoc()): 
                                    ?>
                                        <option value="<?php echo $program['id']; ?>" 
                                                <?php echo $filter_program == $program['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']); ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-auto">
                                <select name="filter_academic_year" class="form-select">
                                    <option value="">All Academic Years</option>
                                    <?php 
                                    $academic_years->data_seek(0);
                                    while ($year = $academic_years->fetch_assoc()): 
                                    ?>
                                        <option value="<?php echo $year['id']; ?>"
                                                <?php echo $filter_academic_year == $year['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($year['year_name']); ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-auto">
                                <button type="submit" class="btn btn-primary">Filter</button>
                                <?php if ($filter_program || $filter_academic_year): ?>
                                    <a href="?" class="btn btn-secondary">Clear</a>
                                <?php endif; ?>
                            </div>
                        </form>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <?php
                        // Build the query for current assignments with filters
                        $where_conditions = ["1=1"];
                        $params = [];
                        $types = "";
                        
                        if ($filter_program) {
                            $where_conditions[] = "p.id = ?";
                            $params[] = $filter_program;
                            $types .= "i";
                        }
                        if ($filter_academic_year) {
                            $where_conditions[] = "pf.academic_year_id = ?";
                            $params[] = $filter_academic_year;
                            $types .= "i";
                        }
                        
                        $where_clause = implode(" AND ", $where_conditions);
                        
                        // Count total records for pagination
                        $count_query = "
                            SELECT COUNT(DISTINCT p.id, pf.academic_year_id) as total
                            FROM programs p
                            INNER JOIN program_fees pf ON p.id = pf.program_id
                            INNER JOIN fee_types ft ON pf.fee_type_id = ft.id AND ft.status = 'active'
                            INNER JOIN academic_years ay ON pf.academic_year_id = ay.id
                            WHERE $where_clause
                        ";
                        
                        $stmt = $conn->prepare($count_query);
                        if (!empty($params)) {
                            $stmt->bind_param($types, ...$params);
                        }
                        $stmt->execute();
                        $total_records = $stmt->get_result()->fetch_assoc()['total'];
                        $total_pages = ceil($total_records / $per_page);
                        
                        // Get current page data
                        $query = "
                            SELECT 
                                p.id as program_id,
                                p.program_code,
                                p.program_name,
                                ay.id as academic_year_id,
                                ay.year_name,
                                GROUP_CONCAT(
                                    CONCAT(ft.name, ' (GHs ', FORMAT(ft.amount, 2), ')')
                                    ORDER BY ft.name
                                    SEPARATOR '<br>'
                                ) as fees,
                                SUM(ft.amount) as total_amount,
                                COUNT(DISTINCT s.id) as student_count,
                                COUNT(ft.id) as fee_count
                            FROM programs p
                            INNER JOIN program_fees pf ON p.id = pf.program_id
                            INNER JOIN fee_types ft ON pf.fee_type_id = ft.id
                            INNER JOIN academic_years ay ON pf.academic_year_id = ay.id
                            LEFT JOIN students s ON s.program_id = p.id AND s.academic_year_id = ay.id
                            WHERE $where_clause AND ft.status = 'active'
                            GROUP BY p.id, pf.academic_year_id
                            ORDER BY ay.year_name DESC, p.program_name
                            LIMIT ? OFFSET ?
                        ";
                        
                        $stmt = $conn->prepare($query);
                        $params[] = $per_page;
                        $params[] = $offset;
                        $types .= "ii";
                        $stmt->bind_param($types, ...$params);
                        $stmt->execute();
                        $assignments = $stmt->get_result();
                        ?>
                        
                        <table class="table align-middle mb-0">
                            <thead>
                                <tr>
                                    <th>Program</th>
                                    <th>Academic Year</th>
                                    <th>Assigned Fees</th>
                                    <th>Total Amount</th>
                                    <th>Students</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($assignments->num_rows > 0): ?>
                                    <?php while ($row = $assignments->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <strong class="text-primary"><?php echo htmlspecialchars($row['program_code']); ?></strong><br>
                                                <small class="text-muted"><?php echo htmlspecialchars($row['program_name']); ?></small>
                                            </td>
                                            <td>
                                                <strong><?php echo htmlspecialchars($row['year_name'] ?? 'N/A'); ?></strong>
                                            </td>
                                            <td>
                                                <?php if ($row['fee_count'] > 0): ?>
                                                    <div class="fees-list">
                                                        <?php echo $row['fees']; ?>
                                                        <div class="mt-1">
                                                            <span class="badge bg-info">
                                                                <?php echo $row['fee_count']; ?> fee<?php echo $row['fee_count'] > 1 ? 's' : ''; ?> assigned
                                                            </span>
                                                        </div>
                                                    </div>
                                                <?php else: ?>
                                                    <span class="text-muted">No fees assigned</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <strong class="text-success">GHs <?php echo number_format($row['total_amount'] ?? 0, 2); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-secondary">
                                                    <?php echo number_format($row['student_count'] ?? 0); ?> student<?php echo $row['student_count'] !== 1 ? 's' : ''; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-danger" 
                                                        onclick="deleteAssignment('<?php echo $row['program_code']; ?>', <?php echo $row['program_id']; ?>, <?php echo $row['academic_year_id']; ?>)">
                                                    <i class="fas fa-trash"></i> Delete
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center py-3">No fee assignments found</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <?php if ($total_pages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <?php if ($page > 1): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?php echo ($page - 1); ?>&filter_program=<?php echo $filter_program; ?>&filter_academic_year=<?php echo $filter_academic_year; ?>">
                                                Previous
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                    
                                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&filter_program=<?php echo $filter_program; ?>&filter_academic_year=<?php echo $filter_academic_year; ?>">
                                                <?php echo $i; ?>
                                            </a>
                                        </li>
                                    <?php endfor; ?>
                                    
                                    <?php if ($page < $total_pages): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?php echo ($page + 1); ?>&filter_program=<?php echo $filter_program; ?>&filter_academic_year=<?php echo $filter_academic_year; ?>">
                                                Next
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Individual Program Assignment -->
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">Assign Fees to Individual Program</h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="assign">
                        
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label class="form-label">Program</label>
                                <select class="form-select" name="program_id" required>
                                    <option value="">Select Program</option>
                                    <?php 
                                    $programs->data_seek(0);
                                    while ($program = $programs->fetch_assoc()): 
                                    ?>
                                    <option value="<?php echo $program['id']; ?>">
                                        <?php echo htmlspecialchars($program['program_name']); ?>
                                        (<?php echo htmlspecialchars($program['program_code']); ?>)
                                    </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Year Level</label>
                                <select class="form-select" name="year_level" id="year_level" required>
                                    <option value="">Select Year Level</option>
                                    <option value="1">1st Year</option>
                                    <option value="2">2nd Year</option>
                                    <option value="3">3rd Year</option>
                                    <option value="4">4th Year</option>
                                    <option value="5">5th Year</option>
                                </select>
                            </div>

                            <div class="col-md-4">
                                <label class="form-label">Academic Year</label>
                                <select class="form-select" name="academic_year_id" required>
                                    <option value="">Select Academic Year</option>
                                    <?php 
                                    $academic_years->data_seek(0);
                                    while ($year = $academic_years->fetch_assoc()): 
                                    ?>
                                    <option value="<?php echo $year['id']; ?>">
                                        <?php echo htmlspecialchars($year['year_name']); ?>
                                    </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-12">
                                <label class="form-label">Select Fees</label>
                                <div class="table-responsive">
                                    <table class="table table-bordered">
                                        <thead>
                                            <tr>
                                                <th style="width: 50px;"></th>
                                                <th>Fee Name</th>
                                                <th>Level</th>
                                                <th>Amount</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $fee_types->data_seek(0);
                                            while ($fee = $fee_types->fetch_assoc()): 
                                            ?>
                                            <tr>
                                                <td>
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="checkbox" 
                                                               name="fee_types[]" 
                                                               value="<?php echo $fee['id']; ?>">
                                                    </div>
                                                </td>
                                                <td><?php echo htmlspecialchars($fee['name']); ?></td>
                                                <td><?php echo htmlspecialchars($fee['description']); ?></td>
                                                <td>GHs <?php echo number_format($fee['amount'], 2); ?></td>
                                            </tr>
                                            <?php endwhile; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-12">
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" name="apply_to_existing" id="applyToExisting">
                                    <label class="form-check-label" for="applyToExisting">
                                        Apply to existing students in this program
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <button type="submit" class="btn btn-primary">
                                    Assign Fees
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Assignment Form (Hidden) -->
<form id="deleteAssignmentForm" method="POST" style="display: none;">
    <input type="hidden" name="action" value="delete_assignment">
    <input type="hidden" name="program_id" id="deleteProgramId">
    <input type="hidden" name="academic_year_id" id="deleteAcademicYearId">
</form>

<script>
function deleteAssignment(programCode, programId, academicYearId) {
    if (confirm('Are you sure you want to delete all fee assignments for ' + programCode + '?\n\nWarning: This will remove all fee assignments for this program in the selected academic year. This action cannot be undone.')) {
        document.getElementById('deleteProgramId').value = programId;
        document.getElementById('deleteAcademicYearId').value = academicYearId;
        document.getElementById('deleteAssignmentForm').submit();
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>


<script>
document.addEventListener('DOMContentLoaded', function() {
    const programSelect = document.querySelector('select[name="program_id"]');
    const academicYearSelect = document.querySelector('select[name="academic_year_id"]');
    const feeCheckboxes = document.querySelectorAll('input[name="fee_types[]"]');
    const programFees = <?php echo json_encode($program_fees); ?>;

    function updateSelectedFees() {
        const programId = programSelect.value;
        const academicYearId = academicYearSelect.value;
        const key = programId + '_' + academicYearId;
        
        // Reset all checkboxes
        feeCheckboxes.forEach(checkbox => {
            checkbox.checked = false;
        });

        // Check boxes based on current assignments
        if (programFees[key]) {
            feeCheckboxes.forEach(checkbox => {
                if (programFees[key].includes(checkbox.value)) {
                    checkbox.checked = true;
                }
            });
        }
    }

    programSelect.addEventListener('change', updateSelectedFees);
    academicYearSelect.addEventListener('change', updateSelectedFees);

    // Bulk assignment helpers
    const selectAllPrograms = document.createElement('button');
    selectAllPrograms.type = 'button';
    selectAllPrograms.className = 'btn btn-sm btn-secondary mb-2';
    selectAllPrograms.textContent = 'Select All Programs';
    document.querySelector('[name="bulk_programs[]"]').closest('.card-body').prepend(selectAllPrograms);

    const selectAllFees = document.createElement('button');
    selectAllFees.type = 'button';
    selectAllFees.className = 'btn btn-sm btn-secondary mb-2';
    selectAllFees.textContent = 'Select All Fees';
    document.querySelector('[name="bulk_fee_types[]"]').closest('.card-body').prepend(selectAllFees);

    selectAllPrograms.onclick = function() {
        const checkboxes = document.querySelectorAll('[name="bulk_programs[]"]');
        const allChecked = Array.from(checkboxes).every(cb => cb.checked);
        checkboxes.forEach(cb => cb.checked = !allChecked);
    };

    selectAllFees.onclick = function() {
        const checkboxes = document.querySelectorAll('[name="bulk_fee_types[]"]');
        const allChecked = Array.from(checkboxes).every(cb => cb.checked);
        checkboxes.forEach(cb => cb.checked = !allChecked);
    };
});
</script>
