<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';
require_once '../includes/database.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Get database connection
$db = Database::getInstance();
$conn = $db->getConnection();

// Get academic years for dropdowns
$academic_years_query = "SELECT * FROM academic_years ORDER BY year_name DESC";
$academic_years_result = $conn->query($academic_years_query);

if (!$academic_years_result) {
    error_log("Academic years query failed: " . $conn->error);
    $_SESSION['error_message'] = "Error loading academic years: " . $conn->error;
    $academic_years = [];
} else {
    error_log("Found " . $academic_years_result->num_rows . " academic years");
    $academic_years = [];
    while ($year = $academic_years_result->fetch_assoc()) {
        $academic_years[] = $year;
    }
}

// Check for messages stored in session
$success_message = '';
$error_message = '';

if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

// Handle program operations (Create, Update, Delete)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $program_code = trim($_POST['program_code'] ?? '');
            $program_name = trim($_POST['program_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $academic_year_id = intval($_POST['academic_year_id'] ?? 0);
            
            if (empty($program_code) || empty($program_name) || empty($academic_year_id)) {
                $_SESSION['error_message'] = "Program code, name and academic year are required.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            // Check if program code already exists
            $check_query = "SELECT id FROM programs WHERE program_code = ?";
            $check_stmt = $conn->prepare($check_query);
            
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("s", $program_code);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "A program with code '" . htmlspecialchars($program_code) . "' already exists.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();

            // Prepare the insert statement
            $insert_query = "INSERT INTO programs (
                program_code, 
                program_name, 
                level,
                academic_year_id,
                status
            ) VALUES (?, ?, ?, ?, 'active')";
            
            $stmt = $conn->prepare($insert_query);
            
            if ($stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            if (!$stmt->bind_param("sssi", $program_code, $program_name, $description, $academic_year_id)) {
                $_SESSION['error_message'] = "Error binding parameters: " . $stmt->error;
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Program created successfully!";
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            } else {
                $_SESSION['error_message'] = "Error creating program: " . $stmt->error;
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            break;
            
        case 'update':
            $program_id = intval($_POST['program_id'] ?? 0);
            $program_code = trim($_POST['program_code'] ?? '');
            $program_name = trim($_POST['program_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $academic_year_id = intval($_POST['academic_year_id'] ?? 0);
            $status = $_POST['status'] ?? 'active';
            
            if ($program_id <= 0 || empty($program_code) || empty($program_name) || $academic_year_id <= 0) {
                $_SESSION['error_message'] = "Invalid program data.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }

            // Check if the program code exists but belongs to a different program
            $check_query = "SELECT id FROM programs WHERE program_code = ? AND id != ?";
            $check_stmt = $conn->prepare($check_query);
            
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("si", $program_code, $program_id);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "A different program with code '" . htmlspecialchars($program_code) . "' already exists.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();
            
            $update_query = "UPDATE programs 
                SET program_code = ?,
                    program_name = ?,
                    description = ?,
                    academic_year_id = ?,
                    status = ?
                WHERE id = ?";
            
            $stmt = $conn->prepare($update_query);
            
            if ($stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            if (!$stmt->bind_param("sssisi", $program_code, $program_name, $description, $academic_year_id, $status, $program_id)) {
                $_SESSION['error_message'] = "Error binding parameters: " . $stmt->error;
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Program updated successfully!";
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            } else {
                $_SESSION['error_message'] = "Error updating program: " . $stmt->error;
                $stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            break;
            
        case 'delete':
            $program_id = intval($_POST['program_id'] ?? 0);
            
            if ($program_id <= 0) {
                $_SESSION['error_message'] = "Invalid program ID.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }

            // First check if the program exists
            $check_query = "SELECT id FROM programs WHERE id = ?";
            $check_stmt = $conn->prepare($check_query);
            
            if ($check_stmt === false) {
                $_SESSION['error_message'] = "Database error: " . $conn->error;
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            
            $check_stmt->bind_param("i", $program_id);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows === 0) {
                $check_stmt->close();
                $_SESSION['error_message'] = "Program not found.";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            $check_stmt->close();
            
            try {
                // First check if there are any students enrolled in this program
                $check_students_query = "SELECT COUNT(*) as student_count FROM students WHERE program_id = ?";
                $check_stmt = $conn->prepare($check_students_query);
                
                if ($check_stmt === false) {
                    throw new Exception("Database error: " . $conn->error);
                }
                
                $check_stmt->bind_param("i", $program_id);
                $check_stmt->execute();
                $result = $check_stmt->get_result();
                $row = $result->fetch_assoc();
                $check_stmt->close();
                
                if ($row['student_count'] > 0) {
                    throw new Exception("Cannot delete program: There are " . $row['student_count'] . " student(s) enrolled in this program. Please transfer or remove these students first.");
                }
                
                // If no students are enrolled, proceed with deletion
                $delete_query = "DELETE FROM programs WHERE id = ?";
                $stmt = $conn->prepare($delete_query);
                
                if ($stmt === false) {
                    throw new Exception("Database error: " . $conn->error);
                }
                
                if (!$stmt->bind_param("i", $program_id)) {
                    throw new Exception("Error binding parameters: " . $stmt->error);
                }
                
                if (!$stmt->execute()) {
                    throw new Exception("Error deleting program: " . $stmt->error);
                }
                
                $stmt->close();
                $_SESSION['success_message'] = "Program deleted successfully!";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
                
            } catch (Exception $e) {
                if (isset($stmt)) {
                    $stmt->close();
                }
                $_SESSION['error_message'] = $e->getMessage();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit();
            }
            break;
    }
}

// Get academic years for dropdown
$academic_years_query = "
    SELECT * 
    FROM academic_years 
    WHERE status = 'active' 
    ORDER BY start_date DESC
";
$academic_years = $conn->query($academic_years_query);

// Get all programs with student counts and academic year info
$programs_query = "
    SELECT 
        p.*,
        ay.year_name,
        ay.id as academic_year_id,
        COALESCE((
            SELECT COUNT(*) 
            FROM students s 
            WHERE s.program_id = p.id
        ), 0) as student_count
    FROM programs p
    LEFT JOIN academic_years ay ON p.academic_year_id = ay.id
    ORDER BY p.program_code
";

error_log("Running programs query: " . $programs_query);
$programs = $conn->query($programs_query);

if ($programs === false) {
    // Create programs table if it doesn't exist
    $create_table = $conn->query("
        CREATE TABLE IF NOT EXISTS programs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            program_code VARCHAR(20) NOT NULL UNIQUE,
            program_name VARCHAR(100) NOT NULL,
            description TEXT,
            academic_year_id INT,
            status ENUM('active', 'inactive') DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (academic_year_id) REFERENCES academic_years(id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    if ($create_table) {
        $programs = $conn->query("
            SELECT 
                p.*,
                ay.year_name,
                (SELECT COUNT(*) FROM students WHERE program_id = p.id) as student_count
            FROM programs p
            LEFT JOIN academic_years ay ON p.academic_year_id = ay.id
            ORDER BY p.program_code
        ");
    }
}

$page_title = "Manage Programs";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Manage Academic Programs</h5>
                    <button type="button" class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addProgramModal">
                        <i class="fas fa-plus"></i> Add New Program
                    </button>
                </div>
                <div class="card-body">
                    <?php 
                    // Debug information
                    if (!$academic_years) {
                        echo '<div class="alert alert-warning">Debug: Academic years query failed</div>';
                    } else {
                        echo '<div class="alert alert-info">Debug: Found ' . $academic_years->num_rows . ' academic years</div>';
                    }
                    
                    if ($success_message) { ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?php echo $success_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php } ?>
                    
                    <?php if ($error_message) { ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?php echo $error_message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php } ?>
                    
                    <div class="table-responsive">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th>Program Code</th>
                                    <th>Program Name</th>
                                    <th>Level</th>
                                    <th>Academic Year</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                error_log("Running programs query: " . $programs_query);
                                
                                if (!$programs) {
                                    error_log("Query failed: " . $conn->error);
                                    echo '<tr><td colspan="7" class="text-center text-danger">Query error: ' . $conn->error . '</td></tr>';
                                } else {
                                    error_log("Query succeeded. Rows found: " . $programs->num_rows);
                                    if ($programs->num_rows > 0) { 
                                        while ($program = $programs->fetch_assoc()) { 
                                ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($program['program_code']); ?></td>
                                            <td>
                                                <?php echo htmlspecialchars($program['program_name']); ?>
                                                <small class="d-block text-muted">
                                                    <?php echo $program['student_count']; ?> student(s)
                                                </small>
                                            </td>
                                            <td><?php echo htmlspecialchars($program['level'] ?? ''); ?></td>
                                            <td><?php echo htmlspecialchars($program['year_name'] ?? 'Not Set'); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $program['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($program['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-info btn-sm" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editProgram<?php echo $program['id']; ?>">
                                                    Edit
                                                </button>
                                                <button type="button" class="btn btn-danger btn-sm"
                                                        onclick="deleteProgram(<?php echo $program['id']; ?>)">
                                                    Delete
                                                </button>
                                            </td>
                                        </tr>

                                        <!-- Edit Program Modal -->
                                        <div class="modal fade" id="editProgram<?php echo $program['id']; ?>" tabindex="-1">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <form method="POST" action="">
                                                        <input type="hidden" name="action" value="update">
                                                        <input type="hidden" name="program_id" value="<?php echo $program['id']; ?>">
                                                        
                                                        <div class="modal-header">
                                                            <h5 class="modal-title">Edit Program</h5>
                                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                        </div>
                                                        
                                                        <div class="modal-body">
                                                            <div class="mb-3">
                                                                <label class="form-label">Academic Year</label>
                                                                <select name="academic_year_id" class="form-select" required>
                                                                    <option value="">Select Academic Year</option>
                                                                    <?php foreach ($academic_years as $year): ?>
                                                                        <option value="<?php echo $year['id']; ?>"
                                                                                <?php echo ($year['id'] == $program['academic_year_id']) ? 'selected' : ''; ?>>
                                                                            <?php echo htmlspecialchars($year['year_name']); ?>
                                                                        </option>
                                                                    <?php endforeach; ?>
                                                                </select>
                                                            </div>

                                                            <div class="mb-3">
                                                                <label class="form-label">Program Code</label>
                                                                <input type="text" name="program_code" class="form-control" 
                                                                       value="<?php echo htmlspecialchars($program['program_code']); ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Program Name</label>
                                                                <input type="text" name="program_name" class="form-control" 
                                                                       value="<?php echo htmlspecialchars($program['program_name']); ?>" required>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Level</label>
                                                                <textarea name="level" class="form-control" rows="2"><?php echo htmlspecialchars($program['level']); ?></textarea>
                                                            </div>
                                                            
                                                            <div class="mb-3">
                                                                <label class="form-label">Status</label>
                                                                <select name="status" class="form-select" required>
                                                                    <option value="active" <?php echo $program['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                                                    <option value="inactive" <?php echo $program['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                                                </select>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                            <button type="submit" class="btn btn-primary">Update Program</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                        <?php 
                                        } // end while
                                    } // end if num_rows > 0
                                    else { 
                                    ?>
                                    <tr>
                                        <td colspan="7" class="text-center">
                                            <p class="my-3">No programs found. Click "Add New Program" to create one.</p>
                                        </td>
                                    </tr>
                                    <?php 
                                    } // end else
                                } // end else after !$programs check 
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Program Modal -->
<div class="modal fade" id="addProgramModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="action" value="create">
                
                <div class="modal-header">
                    <h5 class="modal-title">Add New Program</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Academic Year</label>
                        <select name="academic_year_id" class="form-select" required>
                            <option value="">Select Academic Year</option>
                            <?php 
                            if (!empty($academic_years)) {
                                foreach ($academic_years as $year): 
                            ?>
                                <option value="<?php echo $year['id']; ?>">
                                    <?php echo htmlspecialchars($year['year_name']); ?>
                                </option>
                            <?php 
                                endforeach;
                            } else {
                                error_log("No academic years available");
                                echo '<option value="" disabled>No academic years available</option>';
                            }
                            ?>
                        </select>
                        <?php if (isset($_SESSION['error_message'])): ?>
                            <div class="text-danger small mt-1">
                                <?php echo $_SESSION['error_message']; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Program Code</label>
                        <input type="text" name="program_code" class="form-control" required 
                               placeholder="e.g., BSIT, BSCS">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Program Name</label>
                        <input type="text" name="program_name" class="form-control" required
                               placeholder="e.g., Bachelor of Science in Information Technology">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Level</label>
                        <textarea name="level" class="form-control" rows="2"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Create Program</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Program Form (Hidden) -->
<form id="deleteProgramForm" method="POST" style="display: none;">
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="program_id" id="deleteProgramId">
</form>

<script>
function deleteProgram(programId) {
    if (confirm('Are you sure you want to delete this program? This action cannot be undone.')) {
        document.getElementById('deleteProgramId').value = programId;
        document.getElementById('deleteProgramForm').submit();
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>