<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Pagination setup
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Search functionality
$search_condition = "";
$search_params = [];
$search_types = "";
if (!empty($_GET['search'])) {
    $search = "%" . $_GET['search'] . "%";
    $search_condition = " AND (
        s.student_number LIKE ? OR 
        CONCAT(s.first_name, ' ', s.last_name) LIKE ? OR 
        p.reference_number LIKE ?
    )";
    $search_params = array_fill(0, 3, $search);
    $search_types = "sss";
}

// Filter by date range
$date_condition = "";
$date_params = [];
if (!empty($_GET['start_date']) && !empty($_GET['end_date'])) {
    $date_condition = " AND DATE(p.refund_date) BETWEEN ? AND ?";
    $date_params = [$_GET['start_date'], $_GET['end_date']];
    $search_types .= "ss";
}

// Get refund statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_refunds,
        SUM(refunded_amount) as total_refunded,
        COUNT(CASE WHEN MONTH(refund_date) = MONTH(CURRENT_DATE()) THEN 1 END) as month_refunds,
        SUM(CASE WHEN MONTH(refund_date) = MONTH(CURRENT_DATE()) THEN refunded_amount END) as month_refunded
    FROM payments 
    WHERE status = 'refunded'";
$stats = $conn->query($stats_query)->fetch_assoc();

// Get refunds with search and pagination
$query = "
    SELECT 
        p.*,
        s.student_number,
        s.first_name,
        s.last_name,
        u.email,
        ft.name as fee_type,
        p.processed_by as processor_id,
        (SELECT CONCAT(username) FROM users WHERE id = p.processed_by) as processed_by
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN users u ON s.user_id = u.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    WHERE p.status = 'refunded'" . $search_condition . $date_condition . "
    ORDER BY p.refund_date DESC
    LIMIT ? OFFSET ?";

// Prepare statement with all parameters
$all_params = array_merge($search_params, $date_params, [$per_page, $offset]);
$types = $search_types . "ii";

$stmt = $conn->prepare($query);
if (!empty($all_params)) {
    $stmt->bind_param($types, ...$all_params);
}
$stmt->execute();
$refunds = $stmt->get_result();

// Get total count for pagination
$count_query = "
    SELECT COUNT(*) as total 
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    WHERE p.status = 'refunded'" . $search_condition . $date_condition;

$stmt = $conn->prepare($count_query);
if (!empty($search_params) || !empty($date_params)) {
    $stmt->bind_param(rtrim($search_types, "ii"), ...array_merge($search_params, $date_params));
}
$stmt->execute();
$total_rows = $stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_rows / $per_page);

$page_title = "Refund Management";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Refund Management</h2>
        <a href="dashboard.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back to Dashboard
        </a>
    </div>

    <!-- Statistics Cards -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">Total Refunds</h6>
                    <h2 class="card-title mb-0"><?php echo number_format($stats['total_refunds']); ?></h2>
                    <small class="text-muted">All time</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">Total Amount Refunded</h6>
                    <h2 class="card-title mb-0">GHs <?php echo number_format($stats['total_refunded'], 2); ?></h2>
                    <small class="text-muted">All time</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">This Month's Refunds</h6>
                    <h2 class="card-title mb-0"><?php echo number_format($stats['month_refunds']); ?></h2>
                    <small class="text-muted">Current month</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-subtitle mb-2 text-muted">Month's Refunded Amount</h6>
                    <h2 class="card-title mb-0">GHs <?php echo number_format($stats['month_refunded'], 2); ?></h2>
                    <small class="text-muted">Current month</small>
                </div>
            </div>
        </div>
    </div>

    <!-- Search and Filter -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="get" class="row g-3">
                <div class="col-md-4">
                    <label for="search" class="form-label">Search</label>
                    <input type="text" class="form-control" id="search" name="search" 
                           value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search']) : ''; ?>"
                           placeholder="Student number, name, or reference...">
                </div>
                <div class="col-md-3">
                    <label for="start_date" class="form-label">Start Date</label>
                    <input type="date" class="form-control" id="start_date" name="start_date"
                           value="<?php echo isset($_GET['start_date']) ? htmlspecialchars($_GET['start_date']) : ''; ?>">
                </div>
                <div class="col-md-3">
                    <label for="end_date" class="form-label">End Date</label>
                    <input type="date" class="form-control" id="end_date" name="end_date"
                           value="<?php echo isset($_GET['end_date']) ? htmlspecialchars($_GET['end_date']) : ''; ?>">
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <div class="d-grid gap-2 w-100">
                        <button type="submit" class="btn btn-primary">Search</button>
                        <?php if (!empty($_GET)): ?>
                            <a href="?" class="btn btn-outline-secondary">Clear</a>
                        <?php endif; ?>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Refunds Table -->
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">Refund History</h5>
        </div>
        <div class="table-responsive">
            <table class="table table-hover align-middle mb-0">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Student</th>
                        <th>Reference</th>
                        <th>Fee Type</th>
                        <th class="text-end">Original Amount</th>
                        <th class="text-end">Refunded Amount</th>
                        <th>Reason</th>
                        <th>Processed By</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($refunds->num_rows === 0): ?>
                        <tr>
                            <td colspan="8" class="text-center py-3">No refunds found.</td>
                        </tr>
                    <?php else: ?>
                        <?php while ($refund = $refunds->fetch_assoc()): ?>
                            <tr>
                                <td>
                                    <?php echo date('M d, Y', strtotime($refund['refund_date'])); ?><br>
                                    <small class="text-muted"><?php echo date('g:i A', strtotime($refund['refund_date'])); ?></small>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($refund['student_number']); ?><br>
                                    <small><?php echo htmlspecialchars($refund['first_name'] . ' ' . $refund['last_name']); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($refund['reference_number']); ?></td>
                                <td><?php echo htmlspecialchars($refund['fee_type']); ?></td>
                                <td class="text-end">GHs <?php echo number_format($refund['amount'], 2); ?></td>
                                <td class="text-end">GHs <?php echo number_format($refund['refunded_amount'], 2); ?></td>
                                <td><?php echo nl2br(htmlspecialchars($refund['refund_reason'])); ?></td>
                                <td>
                                    <?php echo htmlspecialchars($refund['processed_by'] ?? 'System'); ?><br>
                                    <small class="text-muted">Accountant</small>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <?php if ($total_pages > 1): ?>
        <div class="card-footer">
            <nav>
                <ul class="pagination justify-content-center mb-0">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?><?php 
                                echo !empty($_GET['search']) ? '&search='.urlencode($_GET['search']) : ''; 
                                echo !empty($_GET['start_date']) ? '&start_date='.urlencode($_GET['start_date']) : '';
                                echo !empty($_GET['end_date']) ? '&end_date='.urlencode($_GET['end_date']) : '';
                            ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                </ul>
            </nav>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>