<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Initialize messages
$success_message = '';
$error_message = '';

if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

// Get database connection
$db = Database::getInstance();
$conn = $db->getConnection();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $db->beginTransaction();

        $action = $_POST['action'] ?? '';

        switch ($action) {
            case 'reset_password':
                $student_id = filter_input(INPUT_POST, 'student_id', FILTER_VALIDATE_INT);
                if (!$student_id) {
                    throw new Exception("Invalid student ID for password reset.");
                }

                // Get student details and verify
                $stmt = $conn->prepare("
                    SELECT u.id as user_id, s.student_number, CONCAT(s.first_name, ' ', s.last_name) as student_name 
                    FROM students s
                    JOIN users u ON s.user_id = u.id 
                    WHERE s.id = ?
                ");
                $stmt->bind_param("i", $student_id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result->num_rows === 0) {
                    throw new Exception("Student not found.");
                }
                
                $student = $result->fetch_assoc();
                
                // Reset password to student number
                $new_password = $student['student_number'];
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

                // Update password
                $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ? AND role = 'student'");
                $stmt->bind_param("si", $hashed_password, $student['user_id']);
                if (!$stmt->execute()) {
                    throw new Exception("Error resetting password.");
                }

                $_SESSION['success_message'] = "Password for " . $student['student_name'] . " has been reset to their student number.";
                break;
        }

        $db->commit();
        
    } catch (Exception $e) {
        $db->rollback();
        $_SESSION['error_message'] = $e->getMessage();
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

$page_title = "Manage Students";
require_once '../includes/header.php';

// Get all students with their program information
$query = "
    SELECT 
        s.id,
        s.student_number,
        s.first_name,
        s.middle_name,
        s.last_name,
        s.year_level,
        s.user_id,
        p.program_code,
        p.program_name,
        u.email,
        u.status as account_status
    FROM students s
    JOIN programs p ON s.program_id = p.id
    JOIN users u ON s.user_id = u.id
    ORDER BY s.last_name, s.first_name
";

$students = $conn->query($query);
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Manage Students</h5>
                    <a href="register_student.php" class="btn btn-light">
                        <i class="fas fa-plus"></i> Add New Student
                    </a>
                </div>
                
                <div class="card-body">
                    <?php if ($success_message): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <?php echo htmlspecialchars($success_message); ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if ($error_message): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <?php echo htmlspecialchars($error_message); ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Student Number</th>
                                    <th>Name</th>
                                    <th>Program</th>
                                    <th>Level</th>
                                    <th>Email</th>
                                    <th>Status</th>
                                    <th class="text-end">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($students && $students->num_rows > 0): ?>
                                    <?php while ($student = $students->fetch_assoc()): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                            <td>
                                                <?php 
                                                echo htmlspecialchars($student['last_name'] . ', ' . $student['first_name']);
                                                if ($student['middle_name']) {
                                                    echo ' ' . htmlspecialchars($student['middle_name']);
                                                }
                                                ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-info">
                                                    <?php echo htmlspecialchars($student['program_code']); ?>
                                                </span>
                                            </td>
                                            <td>Level <?php echo htmlspecialchars($student['year_level']); ?></td>
                                            <td><?php echo htmlspecialchars($student['email']); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $student['account_status'] === 'active' ? 'success' : 'danger'; ?>">
                                                    <?php echo ucfirst($student['account_status']); ?>
                                                </span>
                                            </td>
                                            <td class="text-end">
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="action" value="reset_password">
                                                    <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                    <button type="submit" class="btn btn-warning btn-sm" onclick="return confirm('Reset password for this student? The new password will be their student number.')">
                                                        <i class="fas fa-key"></i>
                                                    </button>
                                                </form>
                                                
                                                <a href="edit_student.php?id=<?php echo $student['id']; ?>" class="btn btn-info btn-sm">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                
                                                <button type="button" class="btn btn-danger btn-sm" onclick="deleteStudent(<?php echo $student['id']; ?>, <?php echo $student['user_id']; ?>)">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="text-center">No students found</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function deleteStudent(studentId, userId) {
    if (confirm('Are you sure you want to delete this student? This cannot be undone.')) {
        let formData = new FormData();
        formData.append('action', 'delete_student');
        formData.append('student_id', studentId);
        formData.append('user_id', userId);

        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            window.location.reload();
        })
        .catch(error => {
            alert('Error deleting student');
            console.error('Error:', error);
        });
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>