<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $username = $conn->real_escape_string($_POST['username']);
                $email = $conn->real_escape_string($_POST['email']);
                $role = $conn->real_escape_string($_POST['role']);
                $password = $_POST['password'];
                
                // Validate password
                if (strlen($password) < 8) {
                    $message = '<div class="alert alert-danger">Password must be at least 8 characters long.</div>';
                    break;
                }
                
                // Check if username exists
                $check = $conn->prepare("SELECT id FROM users WHERE username = ?");
                $check->bind_param("s", $username);
                $check->execute();
                if ($check->get_result()->num_rows > 0) {
                    $message = '<div class="alert alert-danger">Username already exists.</div>';
                    break;
                }
                
                // Check if email exists
                $check = $conn->prepare("SELECT id FROM users WHERE email = ?");
                $check->bind_param("s", $email);
                $check->execute();
                if ($check->get_result()->num_rows > 0) {
                    $message = '<div class="alert alert-danger">Email already exists.</div>';
                    break;
                }
                
                // Create user
                $password_hash = password_hash($password, PASSWORD_DEFAULT);
                $sql = "INSERT INTO users (username, password, email, role, status) VALUES (?, ?, ?, ?, 'active')";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssss", $username, $password_hash, $email, $role);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Administrative user added successfully!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error adding user.</div>';
                }
                break;
                
            case 'reset_password':
                $user_id = intval($_POST['user_id']);
                $new_password = $_POST['new_password'];
                
                // Validate password
                if (strlen($new_password) < 8) {
                    $message = '<div class="alert alert-danger">Password must be at least 8 characters long.</div>';
                    break;
                }
                
                $password_hash = password_hash($new_password, PASSWORD_DEFAULT);
                $sql = "UPDATE users SET password = ? WHERE id = ? AND role IN ('admin', 'accountant')";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("si", $password_hash, $user_id);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Password reset successfully!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error resetting password.</div>';
                }
                break;
                
            case 'toggle_status':
                $user_id = intval($_POST['user_id']);
                $new_status = $_POST['status'] === 'active' ? 'inactive' : 'active';
                
                // Don't allow deactivating own account
                if ($user_id == $_SESSION['user_id']) {
                    $message = '<div class="alert alert-danger">You cannot deactivate your own account.</div>';
                    break;
                }
                
                $sql = "UPDATE users SET status = ? WHERE id = ? AND role IN ('admin', 'accountant')";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("si", $new_status, $user_id);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">User status updated successfully!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error updating user status.</div>';
                }
                break;
        }
    }
}

// Get all administrative users
$users = $conn->query("
    SELECT * FROM users 
    WHERE role IN ('admin', 'accountant') 
    ORDER BY role, username
");

$page_title = "Manage Administrative Users";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Manage Administrative Users</h6>
                        <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addUserModal">
                            Add New User
                        </button>
                    </div>
                </div>
                <div class="card-body px-0 pt-0 pb-2">
                    <?php echo $message; ?>
                    <div class="table-responsive p-0">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th>Username</th>
                                    <th>Role</th>
                                    <th>Email</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($user = $users->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo $user['role'] === 'admin' ? 'primary' : 'info'; ?>">
                                            <?php echo ucfirst($user['role']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo $user['status'] === 'active' ? 'success' : 'danger'; ?>">
                                            <?php echo ucfirst($user['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-warning btn-sm" 
                                                onclick="showResetPassword(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['username']); ?>')">
                                            Reset Password
                                        </button>
                                        <?php if ($user['id'] !== $_SESSION['user_id']): ?>
                                        <form action="" method="POST" class="d-inline">
                                            <input type="hidden" name="action" value="toggle_status">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <input type="hidden" name="status" value="<?php echo $user['status']; ?>">
                                            <button type="submit" class="btn btn-<?php echo $user['status'] === 'active' ? 'danger' : 'success'; ?> btn-sm">
                                                <?php echo $user['status'] === 'active' ? 'Deactivate' : 'Activate'; ?>
                                            </button>
                                        </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add User Modal -->
<div class="modal fade" id="addUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Administrative User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">Email</label>
                        <input type="email" class="form-control" id="email" name="email" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="role" class="form-label">Role</label>
                        <select class="form-control" id="role" name="role" required>
                            <option value="admin">Admin</option>
                            <option value="accountant">Accountant</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" class="form-control" id="password" name="password" 
                               minlength="8" required>
                        <small class="form-text text-muted">
                            Password must be at least 8 characters long.
                        </small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Add User</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Reset Password Modal -->
<div class="modal fade" id="resetPasswordModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Reset Password</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="reset_password">
                    <input type="hidden" name="user_id" id="reset_user_id">
                    
                    <p>Reset password for user: <strong id="reset_username"></strong></p>
                    
                    <div class="mb-3">
                        <label for="new_password" class="form-label">New Password</label>
                        <input type="password" class="form-control" id="new_password" name="new_password" 
                               minlength="8" required>
                        <small class="form-text text-muted">
                            Password must be at least 8 characters long.
                        </small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-warning">Reset Password</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showResetPassword(userId, username) {
    document.getElementById('reset_user_id').value = userId;
    document.getElementById('reset_username').textContent = username;
    new bootstrap.Modal(document.getElementById('resetPasswordModal')).show();
}
</script>

<?php require_once '../includes/footer.php'; ?>