<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

// Start session
session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    http_response_code(403);
    die(json_encode(['error' => 'Access denied']));
}

// Verify database connection
if (!isset($conn) || $conn->connect_error) {
    http_response_code(500);
    die(json_encode(['error' => 'Database connection failed']));
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'reset_password':
                $user_id = intval($_POST['user_id']);
                $student_number = $conn->real_escape_string($_POST['student_number']);
                $password_hash = password_hash($student_number, PASSWORD_DEFAULT);
                
                $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
                if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);
                
                $stmt->bind_param("si", $password_hash, $user_id);
                if (!$stmt->execute()) throw new Exception("Execute failed: " . $stmt->error);
                
                echo json_encode(['success' => true, 'message' => 'Password reset successfully']);
                break;
                
            case 'update_program':
                $student_id = intval($_POST['student_id']);
                $course = $conn->real_escape_string($_POST['course']);
                $year_level = intval($_POST['year_level']);
                
                $stmt = $conn->prepare("UPDATE students SET course = ?, year_level = ? WHERE id = ?");
                if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);
                
                $stmt->bind_param("sii", $course, $year_level, $student_id);
                if (!$stmt->execute()) throw new Exception("Execute failed: " . $stmt->error);
                
                echo json_encode(['success' => true, 'message' => 'Program updated successfully']);
                break;
                
            case 'delete_student':
                $student_id = intval($_POST['student_id']);
                $user_id = intval($_POST['user_id']);
                
                $conn->begin_transaction();
                
                $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
                if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);
                
                $stmt->bind_param("i", $student_id);
                if (!$stmt->execute()) throw new Exception("Failed to delete from students table");
                
                $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
                if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);
                
                $stmt->bind_param("i", $user_id);
                if (!$stmt->execute()) throw new Exception("Failed to delete from users table");
                
                $conn->commit();
                echo json_encode(['success' => true, 'message' => 'Student deleted successfully']);
                break;
                
            default:
                throw new Exception("Invalid action specified");
        }
    } catch (Exception $e) {
        if (isset($conn) && $conn->connect_errno != 0) $conn->rollback();
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}