<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

function getYearLevel($year) {
    $suffixes = ['st', 'nd', 'rd', 'th', 'th'];
    return $year . $suffixes[$year - 1] . ' Year';
}

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';

// Handle student promotion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'promote') {
    if (!isset($_POST['current_year']) || !isset($_POST['program_id'])) {
        $message = '<div class="alert alert-danger">Error: Missing required fields. Please fill in all required fields.</div>';
    } else {
        $current_year = intval($_POST['current_year']);
        $program_id = intval($_POST['program_id']);
        
        try {
            $conn->begin_transaction();

            // Get students to promote
            $stmt = $conn->prepare("
                SELECT s.* 
                FROM students s
                WHERE s.year_level = ? 
                AND s.program_id = ?
            ");
            $stmt->bind_param("ii", $current_year, $program_id);
            $stmt->execute();
            $students = $stmt->get_result();
            
            $promoted_count = 0;
            $skipped_count = 0;
            
            while ($student = $students->fetch_assoc()) {
                if ($student['year_level'] < 5) { // Assuming 5 is the maximum year level
                    // Update student's year level only
                    $new_year = $student['year_level'] + 1;
                    $update_stmt = $conn->prepare("
                        UPDATE students 
                        SET year_level = ? 
                        WHERE id = ?
                    ");
                    $update_stmt->bind_param("ii", $new_year, $student['id']);
                    $update_stmt->execute();
                    
                    // Create a notification for the student
                    $notification_stmt = $conn->prepare("
                        INSERT INTO notifications (
                            user_id, 
                            title, 
                            message, 
                            type
                        ) VALUES (?, ?, ?, 'promotion')
                    ");
                    
                    $title = "Year Level Promotion";
                    $notification_message = "You have been promoted to " . getYearLevel($new_year);
                    
                    $notification_stmt->bind_param(
                        "iss", 
                        $student['user_id'], 
                        $title, 
                        $notification_message
                    );
                    $notification_stmt->execute();
                    
                    $promoted_count++;
                } else {
                    $skipped_count++;
                }
            }
            
            $conn->commit();
            $message = '<div class="alert alert-success">
                Successfully promoted ' . $promoted_count . ' students. 
                Skipped ' . $skipped_count . ' students (maximum year level reached).
            </div>';
            
        } catch (Exception $e) {
            $conn->rollback();
            $message = '<div class="alert alert-danger">Error during promotion: ' . $e->getMessage() . '</div>';
        }
    }
}

// Get available programs
$programs = $conn->query("SELECT id, program_code, program_name FROM programs WHERE status = 'active' ORDER BY program_name");

$page_title = "Student Promotion";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <h6>Student Promotion Management</h6>
                </div>
                <div class="card-body">
                    <?php if ($message): ?>
                        <?php echo $message; ?>
                    <?php endif; ?>
                    
                    <div class="alert alert-info">
                        <h6 class="mb-0"><i class="fas fa-info-circle"></i> Important Notes:</h6>
                        <ul class="mb-0">
                            <li>Students will be promoted to the next year level in their current academic year</li>
                            <li>Maximum year level is 5th year</li>
                            <li>Students will be notified of their promotion</li>
                            <li>This action cannot be undone</li>
                        </ul>
                    </div>
                    
                    <form action="" method="POST" class="mt-4">
                        <input type="hidden" name="action" value="promote">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="program" class="form-label">Program</label>
                                    <select class="form-control" id="program" name="program_id" required>
                                        <option value="">Select Program</option>
                                        <?php while ($program = $programs->fetch_assoc()): ?>
                                            <option value="<?php echo htmlspecialchars($program['id']); ?>">
                                                <?php echo htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']); ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="current_year" class="form-label">Current Year Level</label>
                                    <select class="form-control" id="current_year" name="current_year" required>
                                        <option value="">Select Year Level</option>
                                        <option value="1">1st Year</option>
                                        <option value="2">2nd Year</option>
                                        <option value="3">3rd Year</option>
                                        <option value="4">4th Year</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <button type="submit" class="btn btn-primary" onclick="return confirm('Are you sure you want to promote these students? This action cannot be undone.')">
                                Promote Students
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header pb-0">
                    <h6>Promotion Preview</h6>
                </div>
                <div class="card-body">
                    <div id="preview_section" class="table-responsive">
                        <!-- Preview will be loaded here via AJAX -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Function to load preview of students to be promoted
function loadPreview() {
    const program = document.getElementById('program').value;
    const yearLevel = document.getElementById('current_year').value;
    
    if (program && yearLevel) {
        fetch(`get_promotion_preview.php?program_id=${encodeURIComponent(program)}&year_level=${yearLevel}`)
            .then(response => response.text())
            .then(html => {
                document.getElementById('preview_section').innerHTML = html;
            });
    }
}

// Add event listeners
document.getElementById('program').addEventListener('change', loadPreview);
document.getElementById('current_year').addEventListener('change', loadPreview);
</script>

<?php require_once '../includes/footer.php'; ?>