<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$students = null;

// Get all active programs for the filter
$programs = $conn->query("SELECT * FROM programs WHERE status = 'active' ORDER BY program_code");

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'search' && isset($_POST['program_id']) && isset($_POST['current_year'])) {
            $program_id = intval($_POST['program_id']);
            $current_year = intval($_POST['current_year']);
            
            // Get students of the selected program and year level
            $stmt = $conn->prepare("
                SELECT s.*, p.program_name, p.program_code
                FROM students s
                JOIN programs p ON s.program_id = p.id
                WHERE s.program_id = ? 
                AND s.year_level = ?
                ORDER BY s.last_name, s.first_name
            ");
            $stmt->bind_param('ii', $program_id, $current_year);
            $stmt->execute();
            $students = $stmt->get_result();
            
            if ($students->num_rows === 0) {
                $message = '<div class="alert alert-info">No students found in this year level.</div>';
            }
        } elseif ($_POST['action'] === 'update' && isset($_POST['student_ids']) && is_array($_POST['student_ids'])) {
            try {
                $conn->begin_transaction();
                
                $update_stmt = $conn->prepare("
                    UPDATE students 
                    SET year_level = LEAST(year_level + 1, 4),
                        year_level_personal = LEAST(year_level + 1, 4)
                    WHERE id = ? AND year_level < 4
                ");
                
                $success_count = 0;
                foreach ($_POST['student_ids'] as $student_id) {
                    $update_stmt->bind_param('i', $student_id);
                    if ($update_stmt->execute() && $update_stmt->affected_rows > 0) {
                        $success_count++;
                    }
                }
                
                $conn->commit();
                $message = '<div class="alert alert-success">Successfully updated ' . $success_count . ' student(s) to the next year level.</div>';
                
            } catch (Exception $e) {
                $conn->rollback();
                $message = '<div class="alert alert-danger">Error: ' . $e->getMessage() . '</div>';
            }
        }
    }
}

$page_title = "Update Student Year Level";
require_once '../includes/header.php';
?>

<div class="container mt-4">
    <h2>Update Student Year Level</h2>
    
    <?php echo $message; ?>

    <div class="card">
        <div class="card-body">
            <form method="post" class="mb-4">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="program_id">Program:</label>
                            <select name="program_id" id="program_id" class="form-control" required>
                                <option value="">Select Program</option>
                                <?php while ($program = $programs->fetch_assoc()): ?>
                                    <option value="<?= $program['id'] ?>" <?= isset($_POST['program_id']) && $_POST['program_id'] == $program['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']) ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="current_year">Current Year Level:</label>
                            <select name="current_year" id="current_year" class="form-control" required>
                                <option value="">Select Year Level</option>
                                <option value="1" <?= isset($_POST['current_year']) && $_POST['current_year'] == '1' ? 'selected' : '' ?>>1st Year</option>
                                <option value="2" <?= isset($_POST['current_year']) && $_POST['current_year'] == '2' ? 'selected' : '' ?>>2nd Year</option>
                                <option value="3" <?= isset($_POST['current_year']) && $_POST['current_year'] == '3' ? 'selected' : '' ?>>3rd Year</option>
                                <option value="4" <?= isset($_POST['current_year']) && $_POST['current_year'] == '4' ? 'selected' : '' ?>>4th Year</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="mt-3">
                    <button type="submit" name="action" value="search" class="btn btn-primary">
                        <i class="fas fa-search"></i> Search Students
                    </button>
                </div>
            </form>

            <?php if ($students && $students->num_rows > 0): ?>
                <form method="post">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all"></th>
                                    <th>Student ID</th>
                                    <th>Name</th>
                                    <th>Current Year Level</th>
                                    <th>Next Year Level</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($student = $students->fetch_assoc()): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="student_ids[]" value="<?= $student['id'] ?>" class="student-checkbox">
                                        </td>
                                        <td><?= htmlspecialchars($student['student_number']) ?></td>
                                        <td><?= htmlspecialchars($student['last_name'] . ', ' . $student['first_name']) ?></td>
                                        <td><?= $student['year_level'] ?> Year</td>
                                        <td><?= ($student['year_level'] + 1) ?> Year</td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <div class="mt-3">
                        <button type="submit" name="action" value="update" class="btn btn-success" id="update-btn" disabled>
                            <i class="fas fa-arrow-up"></i> Update Selected Students
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const selectAll = document.getElementById('select-all');
    const studentCheckboxes = document.getElementsByClassName('student-checkbox');
    const updateBtn = document.getElementById('update-btn');

    // Select all checkbox functionality
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            Array.from(studentCheckboxes).forEach(checkbox => {
                checkbox.checked = this.checked;
            });
            updateButtonState();
        });
    }

    // Individual checkbox functionality
    Array.from(studentCheckboxes).forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateButtonState();
            
            // Update select-all checkbox state
            if (selectAll) {
                selectAll.checked = Array.from(studentCheckboxes).every(cb => cb.checked);
            }
        });
    });

    // Update button state based on selections
    function updateButtonState() {
        if (updateBtn) {
            updateBtn.disabled = !Array.from(studentCheckboxes).some(cb => cb.checked);
        }
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>