<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify admin access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';

// Get filter values
$filter_program = isset($_GET['filter_program']) ? $_GET['filter_program'] : '';
$filter_year_level = isset($_GET['filter_year_level']) ? (int)$_GET['filter_year_level'] : 0;
$filter_academic_year = isset($_GET['filter_academic_year']) ? (int)$_GET['filter_academic_year'] : 0;

// Build base query
$query = "SELECT 
            s.*, 
            p.program_name, 
            p.program_code, 
            u.created_at, 
            u.id as user_id,
            COALESCE(ay.year_name, 'Not Set') as academic_year,
            COALESCE(ay.id, 0) as academic_year_id
         FROM students s 
         JOIN users u ON s.user_id = u.id 
         JOIN programs p ON s.program_id = p.id 
         LEFT JOIN academic_years ay ON s.academic_year_id = ay.id
         WHERE 1=1";

$params = array();
$types = "";

// Add filters
if (!empty($filter_program)) {
    $query .= " AND p.program_code = ?";
    $params[] = $filter_program;
    $types .= "s";
}

if ($filter_year_level > 0) {
    $query .= " AND s.year_level = ?";
    $params[] = $filter_year_level;
    $types .= "i";
}

if ($filter_academic_year > 0) {
    $query .= " AND EXISTS (
        SELECT 1 FROM student_fees sf2 
        WHERE sf2.student_id = s.id 
        AND sf2.academic_year_id = ?
    )";
    $params[] = $filter_academic_year;
    $types .= "i";
}

// Add group by and order
$query .= " GROUP BY s.id ORDER BY u.created_at DESC LIMIT 50";

                                // Build query with filters
                                $query = "SELECT s.*, p.program_name, p.program_code, u.created_at, u.id as user_id,
                                         COALESCE(ay.year_name, 
                                            (SELECT year_name FROM academic_years WHERE is_current = 1 LIMIT 1)
                                         ) as academic_year 
                                         FROM students s 
                                         JOIN users u ON s.user_id = u.id 
                                         JOIN programs p ON s.program_id = p.id 
                                         LEFT JOIN academic_years ay ON ay.id = (
                                             SELECT academic_year_id 
                                             FROM student_fees 
                                             WHERE student_id = s.id 
                                             ORDER BY created_at DESC 
                                             LIMIT 1
                                         )
                                         WHERE 1=1";

                                $params = array();
                                $types = "";

                                if (!empty($filter_program)) {
                                    $query .= " AND p.program_code = ?";
                                    $params[] = $filter_program;
                                    $types .= "s";
                                }
                                if ($filter_year_level > 0) {
                                    $query .= " AND s.year_level = ?";
                                    $params[] = $filter_year_level;
                                    $types .= "i";
                                }
                                if ($filter_academic_year > 0) {
                                    $query .= " AND s.academic_year_id = ?";
                                    $params[] = $filter_academic_year;
                                    $types .= "i";
                                }

                                $query .= " ORDER BY u.created_at DESC LIMIT 50";

                                $stmt = $conn->prepare($query);
                                if (!empty($params)) {
                                    $stmt->bind_param($types, ...$params);
                                }
                                $stmt->execute();
                                $result = $stmt->get_result();                     $filter_program = isset($_GET['filter_program']) ? $_GET['filter_program'] : '';
                                $filter_year_level = isset($_GET['filter_year_level']) ? (int)$_GET['filter_year_level'] : 0;
                                $filter_academic_year = isset($_GET['filter_academic_year']) ? (int)$_GET['filter_academic_year'] : 0;

                                // Build query with filters
                                $query = "SELECT DISTINCT
                                            s.*, 
                                            p.program_name, 
                                            p.program_code, 
                                            u.created_at, 
                                            u.id as user_id,
                                            u.email,
                                            ay.year_name as academic_year
                                         FROM students s 
                                         JOIN users u ON s.user_id = u.id 
                                         JOIN programs p ON s.program_id = p.id 
                                         JOIN academic_years ay ON ay.id = s.academic_year_id
                                         WHERE 1=1";

                                $params = array();
                                $types = "";

                                if (!empty($filter_program)) {
                                    $query .= " AND p.program_code = ?";
                                    $params[] = $filter_program;
                                    $types .= "s";
                                }
                                if ($filter_year_level > 0) {
                                    $query .= " AND s.year_level = ?";
                                    $params[] = $filter_year_level;
                                    $types .= "i";
                                }
                                if ($filter_academic_year > 0) {
                                    $query .= " AND s.academic_year_id = ?";
                                    $params[] = $filter_academic_year;
                                    $types .= "i";
                                }

                                $query .= " ORDER BY u.created_at DESC LIMIT 50";
                                
                                $stmt = $conn->prepare($query);
                                if (!empty($params)) {
                                    $stmt->bind_param($types, ...$params);
                                }
                                $stmt->execute();
                                $result = $stmt->get_result();

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    switch ($_POST['action']) {
            case 'edit':
                try {
                    $student_id = intval($_POST['student_id']);
                    $program_code = $conn->real_escape_string($_POST['program']);
                    
                    // Get program ID
                    $stmt = $conn->prepare("SELECT id FROM programs WHERE program_code = ? AND status = 'active'");
                    $stmt->bind_param("s", $program_code);
                    $stmt->execute();
                    $program_result = $stmt->get_result();
                    $program_data = $program_result->fetch_assoc();
                    $program_id = $program_data['id'];
                    
                    // Begin transaction
                    $conn->begin_transaction();
                    
                    // Update student record
                    $stmt = $conn->prepare("
                        UPDATE students SET 
                            student_number = ?,
                            first_name = ?,
                            last_name = ?,
                            program_id = ?,
                            year_level = ?,
                            contact_number = ?,
                            address = ?,
                            academic_year_id = ?
                        WHERE id = ?
                    ");
                    $stmt->bind_param(
                        "sssiissii",
                        $_POST['student_number'],
                        $_POST['first_name'],
                        $_POST['last_name'],
                        $program_id,
                        $_POST['year_level'],
                        $_POST['contact_number'],
                        $_POST['address'],
                        $_POST['academic_year'],
                        $student_id
                    );
                    $stmt->execute();
                    
                    // Update user email
                    $stmt = $conn->prepare("
                        UPDATE users u 
                        JOIN students s ON u.id = s.user_id 
                        SET u.email = ?, u.username = ? 
                        WHERE s.id = ?
                    ");
                    $stmt->bind_param("ssi", $_POST['email'], $_POST['student_number'], $student_id);
                    $stmt->execute();
                    
                    $conn->commit();
                    $message = '<div class="alert alert-success">Student updated successfully!</div>';
                } catch (Exception $e) {
                    $conn->rollback();
                    $message = '<div class="alert alert-danger">Error updating student: ' . $e->getMessage() . '</div>';
                }
                break;
                
            case 'delete':
                try {
                    $student_id = intval($_POST['student_id']);
                    $user_id = intval($_POST['user_id']);
                    
                    $conn->begin_transaction();
                    
                    // First delete all student_fees records
                    $stmt = $conn->prepare("DELETE FROM student_fees WHERE student_id = ?");
                    $stmt->bind_param("i", $student_id);
                    $stmt->execute();
                    
                    // Then delete student record
                    $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
                    $stmt->bind_param("i", $student_id);
                    $stmt->execute();
                    
                    // Finally delete user account
                    $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
                    $stmt->bind_param("i", $user_id);
                    $stmt->execute();
                    
                    $conn->commit();
                    $message = '<div class="alert alert-success">Student deleted successfully!</div>';
                } catch (Exception $e) {
                    $conn->rollback();
                    $message = '<div class="alert alert-danger">Error deleting student: ' . $e->getMessage() . '</div>';
                }
                break;

            case 'bulk_delete':
                try {
                    if (!isset($_POST['student_ids']) || !isset($_POST['user_ids'])) {
                        throw new Exception('No students selected for deletion');
                    }

                    $student_ids = array_map('intval', explode(',', $_POST['student_ids']));
                    $user_ids = array_map('intval', explode(',', $_POST['user_ids']));

                    $conn->begin_transaction();

                    // First delete all student_fees records
                    $student_ids_str = implode(',', $student_ids);
                    $stmt = $conn->prepare("DELETE FROM student_fees WHERE student_id IN ($student_ids_str)");
                    $stmt->execute();

                    // Then delete student records
                    $stmt = $conn->prepare("DELETE FROM students WHERE id IN ($student_ids_str)");
                    $stmt->execute();

                    // Finally delete user accounts
                    $user_ids_str = implode(',', $user_ids);
                    $stmt = $conn->prepare("DELETE FROM users WHERE id IN ($user_ids_str)");
                    $stmt->execute();

                    $conn->commit();
                    $message = '<div class="alert alert-success">' . count($student_ids) . ' students deleted successfully!</div>';
                } catch (Exception $e) {
                    $conn->rollback();
                    $message = '<div class="alert alert-danger">Error deleting students: ' . $e->getMessage() . '</div>';
                }
                break;
                
            case 'upload':
                if (isset($_FILES['student_file']) && $_FILES['student_file']['error'] === UPLOAD_ERR_OK) {
            $program = $conn->real_escape_string($_POST['program']);
            $year_level = intval($_POST['year_level']);
            $academic_year_id = intval($_POST['academic_year']);
            
            // Debug information
            error_log("Upload started with:");
            error_log("Program: " . $program);
            error_log("Year Level: " . $year_level);
            error_log("Academic Year ID: " . $academic_year_id);
            
            // Validate academic year exists
            $academic_check = $conn->prepare("SELECT id FROM academic_years WHERE id = ?");
            $academic_check->bind_param("i", $academic_year_id);
            $academic_check->execute();
            if ($academic_check->get_result()->num_rows === 0) {
                $message = '<div class="alert alert-danger">Invalid academic year selected.</div>';
                $academic_check->close();
                break;
            }
            $academic_check->close();
            
            // Validate academic year exists
            $academic_check = $conn->prepare("SELECT id FROM academic_years WHERE id = ?");
            $academic_check->bind_param("i", $academic_year_id);
            $academic_check->execute();
            $academic_result = $academic_check->get_result();
            
            if ($academic_result->num_rows === 0) {
                $message = '<div class="alert alert-danger">Invalid academic year selected.</div>';
                $academic_check->close();
                break;
            }
            $academic_check->close();
            
            // Validate program exists
            $program_check = $conn->prepare("SELECT id FROM programs WHERE program_code = ? AND status = 'active'");
            $program_check->bind_param("s", $program);
            $program_check->execute();
            $result = $program_check->get_result();
            
            if ($result->num_rows === 0) {
                $message = '<div class="alert alert-danger">Invalid program selected.</div>';
                $program_check->close();
            } else {
                $program_check->close();
                
                // Open uploaded file
                $handle = fopen($_FILES['student_file']['tmp_name'], "r");
                
                if ($handle !== FALSE) {
                    // Skip header row
                    $header = fgetcsv($handle);
                    
                    // Begin transaction
                    $conn->begin_transaction();
                    
                    try {
                        while (($data = fgetcsv($handle)) !== FALSE) {
                            $student_number = $conn->real_escape_string($data[0]);
                            $first_name = $conn->real_escape_string($data[1]);
                            $last_name = $conn->real_escape_string($data[2]);
                            $email = $conn->real_escape_string($data[3]);
                            $contact = $conn->real_escape_string($data[4]);
                            $address = $conn->real_escape_string($data[5]);
                            
                            // Check if student number already exists
                            $check_sql = "SELECT id FROM users WHERE username = ?";
                            $check_stmt = $conn->prepare($check_sql);
                            $check_stmt->bind_param("s", $student_number);
                            $check_stmt->execute();
                            $check_result = $check_stmt->get_result();
                            
                            if ($check_result->num_rows > 0) {
                                throw new Exception("Student number '" . htmlspecialchars($student_number) . "' already exists. Please check your data and try again.");
                            }
                            $check_stmt->close();
                            
                            // Create user account
                            $password_hash = password_hash($student_number, PASSWORD_DEFAULT);
                            
                            $sql = "INSERT INTO users (username, password, email, role, status) VALUES (?, ?, ?, 'student', 'active')";
                            $stmt = $conn->prepare($sql);
                            $stmt->bind_param("sss", $student_number, $password_hash, $email);
                            $stmt->execute();
                            
                            $user_id = $conn->insert_id;
                            
                            // Get program ID from program code
                            $program_sql = "SELECT id FROM programs WHERE program_code = ? AND status = 'active'";
                            $program_stmt = $conn->prepare($program_sql);
                            $program_stmt->bind_param("s", $program);
                            $program_stmt->execute();
                            $program_result = $program_stmt->get_result();
                            $program_data = $program_result->fetch_assoc();
                            $program_id = $program_data['id'];
                            $program_stmt->close();
                            
                            // Log academic year ID for debugging
                            error_log("Uploading student with academic_year_id: " . $academic_year_id);
                            
                            // Create student record
                            $sql = "INSERT INTO students (user_id, student_number, first_name, last_name, program_id, year_level, contact_number, address, academic_year_id) 
                                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                            $stmt = $conn->prepare($sql);
                            $stmt->bind_param("isssisssi", $user_id, $student_number, $first_name, $last_name, $program_id, $year_level, $contact, $address, $academic_year_id);
                            
                            if (!$stmt->execute()) {
                                error_log("Error inserting student: " . $stmt->error);
                                throw new Exception("Error inserting student: " . $stmt->error);
                            }
                            
                            // Log success
                            error_log("Successfully inserted student with ID: " . $conn->insert_id);
                            
                            // Just create student record with academic year
                            $student_id = $conn->insert_id;
                            
                            // Get a valid fee type ID (using the first active fee type)
                            $fee_type_sql = "SELECT id FROM fee_types WHERE status = 'active' LIMIT 1";
                            $fee_type_result = $conn->query($fee_type_sql);
                            $fee_type_data = $fee_type_result->fetch_assoc();
                            $fee_type_id = $fee_type_data['id'];

                            // Create initial student fees record with a valid fee type
                            $fees_sql = "INSERT INTO student_fees (student_id, academic_year_id, fee_type_id, created_at) 
                                       VALUES (?, ?, ?, NOW())";
                            $fees_stmt = $conn->prepare($fees_sql);
                            $fees_stmt->bind_param("iii", $student_id, $academic_year_id, $fee_type_id);
                            $fees_stmt->execute();
                        }
                        
                        $conn->commit();
                        $message = '<div class="alert alert-success">Students uploaded successfully!</div>';
                    } catch (Exception $e) {
                        $conn->rollback();
                        $message = '<div class="alert alert-danger">Error uploading students: ' . $e->getMessage() . '</div>';
                    }
                    
                    fclose($handle);
                }
            }
                break;
        }
    }
}

// Get available programs
$programs = $conn->query("
    SELECT id, program_code, program_name 
    FROM programs 
    WHERE status = 'active' 
    ORDER BY program_name
");

// Get academic years
$academic_years = $conn->query("
    SELECT id, year_name, is_current 
    FROM academic_years 
    ORDER BY is_current DESC, year_name DESC
");

$page_title = "Upload Students";
require_once '../includes/header.php';
?>
<style>
    .form-check-input {
        cursor: pointer;
    }
    #deleteSelectedBtn {
        transition: opacity 0.3s;
    }
    .checkbox-column {
        width: 40px;
        text-align: center;
    }
</style>
<?php
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Upload Students</h6>
                        <button type="button" class="btn btn-danger btn-sm" id="deleteSelectedBtn" style="display: none;">
                            <i class="fas fa-trash"></i> Delete Selected Students
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <?php echo $message; ?>
                    
                    <form action="" method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="upload">
                        
                        <div class="mb-3">
                            <label for="academic_year" class="form-label">Academic Year</label>
                            <select class="form-select" id="academic_year" name="academic_year" required>
                                <option value="">Select Academic Year</option>
                                <?php while ($year = $academic_years->fetch_assoc()): ?>
                                    <option value="<?php echo $year['id']; ?>"<?php echo (isset($year['is_current']) && $year['is_current'] == 1) ? ' selected' : ''; ?>>
                                        <?php echo htmlspecialchars($year['year_name']); ?><?php echo (isset($year['is_current']) && $year['is_current'] == 1) ? ' (Current)' : ''; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="program" class="form-label">Program</label>
                            <select class="form-select" id="program" name="program" required>
                                <option value="">Select Program</option>
                                <?php 
                                $programs->data_seek(0);
                                while ($program = $programs->fetch_assoc()): 
                                ?>
                                    <option value="<?php echo htmlspecialchars($program['program_code']); ?>">
                                        <?php echo htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="year_level" class="form-label">Year Level</label>
                            <select class="form-select" id="year_level" name="year_level" required>
                                <option value="1">1st Year</option>
                                <option value="2">2nd Year</option>
                                <option value="3">3rd Year</option>
                                <option value="4">4th Year</option>
                                <option value="5">5th Year</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="student_file" class="form-label">Student CSV File</label>
                            <input type="file" class="form-control" id="student_file" name="student_file" accept=".csv" required>
                            <small class="form-text text-muted">
                                CSV file should have the following columns:<br>
                                Student Number, First Name, Last Name, Email, Contact Number, Address
                            </small>
                        </div>
                        
                        <div class="alert alert-info">
                            <strong>Note:</strong> Initial password for each student will be their student number.
                        </div>
                        
                        <button type="submit" class="btn btn-primary">Upload Students</button>
                    </form>
                </div>
            </div>
            
            <!-- Sample CSV Template -->
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <h6>CSV Template</h6>
                </div>
                <div class="card-body">
                    <p>Download this sample CSV template:</p>
                    <pre>Student Number,First Name,Last Name,Email,Contact Number,Address
2025001,John,Doe,john.doe@example.com,1234567890,123 Main St
2025002,Jane,Smith,jane.smith@example.com,0987654321,456 Oak Ave</pre>
                    <a href="sample_students.csv" class="btn btn-secondary" download>Download Template</a>
                </div>
            </div>

            <!-- Recently Uploaded Students -->
            <div class="card">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">Recently Uploaded Students</h6>
                        <button type="button" class="btn btn-danger btn-sm ms-2" id="deleteSelectedBtn" style="display: none; opacity: 1;">
                            <i class="fas fa-trash-alt me-1"></i> Delete Selected (<span id="selectedCount">0</span>)
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Filter Form -->
                    <form method="GET" class="mb-4">
                        <div class="row g-3">
                            <div class="col-md-3">
                                <label for="filter_program" class="form-label">Program</label>
                                <select class="form-select" id="filter_program" name="filter_program">
                                    <option value="">All Programs</option>
                                    <?php 
                                    $programs->data_seek(0);
                                    while ($program = $programs->fetch_assoc()): 
                                    ?>
                                        <option value="<?php echo htmlspecialchars($program['program_code']); ?>"
                                            <?php echo (isset($_GET['filter_program']) && $_GET['filter_program'] == $program['program_code']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']); ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="filter_year_level" class="form-label">Year Level</label>
                                <select class="form-select" id="filter_year_level" name="filter_year_level">
                                    <option value="">All Years</option>
                                    <?php for($i = 1; $i <= 5; $i++): ?>
                                        <option value="<?php echo $i; ?>" <?php echo (isset($_GET['filter_year_level']) && $_GET['filter_year_level'] == $i) ? 'selected' : ''; ?>>
                                            <?php echo $i; ?>st Year
                                        </option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="filter_academic_year" class="form-label">Academic Year</label>
                                <select class="form-select" id="filter_academic_year" name="filter_academic_year">
                                    <option value="">All Academic Years</option>
                                    <?php 
                                    $academic_years->data_seek(0);
                                    while ($year = $academic_years->fetch_assoc()): 
                                    ?>
                                        <option value="<?php echo $year['id']; ?>"
                                            <?php echo (isset($_GET['filter_academic_year']) && $_GET['filter_academic_year'] == $year['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($year['year_name']); ?>
                                            <?php echo (isset($year['is_current']) && $year['is_current'] == 1) ? ' (Current)' : ''; ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary">Filter</button>
                                </div>
                            </div>
                        </div>
                    </form>

                    <div class="table-responsive p-0">
                        <table class="table align-items-center mb-0">
                            <thead>
                                <tr>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 checkbox-column">
                                        <div class="form-check d-flex justify-content-center">
                                            <input class="form-check-input" type="checkbox" id="selectAll">
                                        </div>
                                    </th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7">Student Name</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Student Number</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Program</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Year Level</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Academic Year</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Contact</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Date Added</th>
                                    <th class="text-uppercase text-secondary text-xxs font-weight-bolder opacity-7 ps-2">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                // Get recently uploaded students
                                $query = "SELECT s.*, 
                                         p.program_name, 
                                         p.program_code, 
                                         u.created_at, 
                                         u.id as user_id, 
                                         u.email,
                                         ay.year_name as academic_year,
                                         ay.id as academic_year_id
                                         FROM students s 
                                         JOIN users u ON s.user_id = u.id 
                                         JOIN programs p ON s.program_id = p.id 
                                         JOIN academic_years ay ON s.academic_year_id = ay.id
                                         ORDER BY u.created_at DESC 
                                         LIMIT 10";
                                $result = $conn->query($query);
                                
                                while ($student = $result->fetch_assoc()):
                                ?>
                                <tr>
                                    <td class="checkbox-column">
                                        <div class="form-check d-flex justify-content-center">
                                            <input class="form-check-input student-checkbox" 
                                                   type="checkbox" 
                                                   value="<?php echo $student['id']; ?>" 
                                                   data-user-id="<?php echo $student['user_id']; ?>">
                                        </div>
                                    </td>
                                    <td>
                                        <div class="d-flex px-2 py-1">
                                            <div class="d-flex flex-column justify-content-center">
                                                <h6 class="mb-0 text-sm"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></h6>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <p class="text-xs font-weight-bold mb-0"><?php echo htmlspecialchars($student['student_number']); ?></p>
                                    </td>
                                    <td>
                                        <p class="text-xs font-weight-bold mb-0"><?php echo htmlspecialchars($student['program_name']); ?></p>
                                    </td>
                                    <td>
                                        <p class="text-xs font-weight-bold mb-0"><?php echo htmlspecialchars($student['year_level']); ?></p>
                                    </td>
                                    <td>
                                        <p class="text-xs font-weight-bold mb-0"><?php echo htmlspecialchars($student['academic_year']); ?></p>
                                    </td>
                                    <td>
                                        <p class="text-xs text-secondary mb-0"><?php echo htmlspecialchars($student['contact_number']); ?></p>
                                    </td>
                                    <td>
                                        <span class="text-secondary text-xs font-weight-bold">
                                            <?php echo date('M d, Y', strtotime($student['created_at'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php 
                                            $studentData = array(
                                                'id' => $student['id'],
                                                'studentNumber' => $student['student_number'],
                                                'firstName' => $student['first_name'],
                                                'lastName' => $student['last_name'],
                                                'email' => isset($student['email']) ? $student['email'] : $student['student_number'] . '@example.com',
                                                'contact' => $student['contact_number'],
                                                'address' => $student['address'],
                                                'program' => $student['program_code'],
                                                'year' => $student['year_level'],
                                                'academicYear' => $student['academic_year_id']
                                            );
                                            $dataAttributes = '';
                                            foreach ($studentData as $key => $value) {
                                                $dataAttributes .= ' data-' . strtolower(preg_replace('/([A-Z])/', '-$1', $key)) . '="' . htmlspecialchars($value) . '"';
                                            }
                                        ?>
                                        <button type="button" 
                                                class="btn btn-sm btn-info edit-student"
                                                data-bs-toggle="modal"
                                                data-bs-target="#editStudentModal"
                                                <?php echo $dataAttributes; ?>>
                                            <i class="fas fa-edit"></i> Edit
                                        </button>
                                        <button type="button" 
                                                class="btn btn-sm btn-danger delete-student"
                                                data-bs-toggle="modal"
                                                data-bs-target="#deleteStudentModal"
                                                data-id="<?php echo $student['id']; ?>"
                                                data-user-id="<?php echo $student['user_id']; ?>"
                                                data-name="<?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?>">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" role="dialog" aria-labelledby="bulkDeleteModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="bulkDeleteModalLabel">Delete Selected Students</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="bulkDeleteForm" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="bulk_delete">
                    <input type="hidden" name="student_ids" id="bulk_delete_student_ids">
                    <input type="hidden" name="user_ids" id="bulk_delete_user_ids">
                    <p>Are you sure you want to delete the selected students?</p>
                    <p>Number of students to delete: <span id="delete_count">0</span></p>
                    <p class="text-danger">This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Selected</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Student Modal -->
<div class="modal fade" id="editStudentModal" tabindex="-1" role="dialog" aria-labelledby="editStudentModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editStudentModalLabel">Edit Student</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="editStudentForm" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="student_id" id="edit_student_id">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_student_number" class="form-label">Student Number</label>
                            <input type="text" class="form-control" id="edit_student_number" name="student_number" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_program" class="form-label">Program</label>
                            <select class="form-select" id="edit_program" name="program" required>
                                <?php 
                                $programs->data_seek(0);
                                while ($program = $programs->fetch_assoc()): 
                                ?>
                                    <option value="<?php echo htmlspecialchars($program['program_code']); ?>">
                                        <?php echo htmlspecialchars($program['program_code'] . ' - ' . $program['program_name']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_first_name" class="form-label">First Name</label>
                            <input type="text" class="form-control" id="edit_first_name" name="first_name" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_last_name" class="form-label">Last Name</label>
                            <input type="text" class="form-control" id="edit_last_name" name="last_name" required>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="edit_email" name="email" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_year_level" class="form-label">Year Level</label>
                            <select class="form-select" id="edit_year_level" name="year_level" required>
                                <option value="1">1st Year</option>
                                <option value="2">2nd Year</option>
                                <option value="3">3rd Year</option>
                                <option value="4">4th Year</option>
                                <option value="5">5th Year</option>
                            </select>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label for="edit_contact" class="form-label">Contact Number</label>
                            <input type="text" class="form-control" id="edit_contact" name="contact_number">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="edit_address" class="form-label">Address</label>
                            <input type="text" class="form-control" id="edit_address" name="address">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="edit_academic_year" class="form-label">Academic Year</label>
                            <select class="form-select" id="edit_academic_year" name="academic_year" required>
                                <?php 
                                $academic_years->data_seek(0);
                                while ($year = $academic_years->fetch_assoc()): 
                                ?>
                                    <option value="<?php echo $year['id']; ?>">
                                        <?php echo htmlspecialchars($year['year_name']); ?>
                                        <?php echo (isset($year['is_current']) && $year['is_current'] == 1) ? ' (Current)' : ''; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteStudentModal" tabindex="-1" role="dialog" aria-labelledby="deleteStudentModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteStudentModalLabel">Delete Student</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="deleteStudentForm" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="student_id" id="delete_student_id">
                    <input type="hidden" name="user_id" id="delete_user_id">
                    <p>Are you sure you want to delete <span id="delete_student_name"></span>?</p>
                    <p class="text-danger">This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" role="dialog" aria-labelledby="bulkDeleteModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="bulkDeleteModalLabel">Delete Selected Students</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="bulkDeleteForm" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="bulk_delete">
                    <input type="hidden" name="student_ids" id="bulk_delete_student_ids">
                    <input type="hidden" name="user_ids" id="bulk_delete_user_ids">
                    <p>Are you sure you want to delete the selected students?</p>
                    <p>Number of students to delete: <span id="delete_count"></span></p>
                    <p class="text-danger">This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Selected</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Checkboxes and bulk delete functionality
    const selectAll = document.getElementById('selectAll');
    const studentCheckboxes = document.querySelectorAll('.student-checkbox');
    const deleteSelectedBtn = document.getElementById('deleteSelectedBtn');
    const bulkDeleteModal = new bootstrap.Modal(document.getElementById('bulkDeleteModal'));

    // Handle select all checkbox
    selectAll.addEventListener('change', function() {
        studentCheckboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateDeleteButton();
    });

    // Handle individual checkboxes
    studentCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateDeleteButton();
            // Update select all checkbox
            selectAll.checked = [...studentCheckboxes].every(cb => cb.checked);
        });
    });

    // Update delete button visibility and selected count
    function updateDeleteButton() {
        const checkedCount = [...studentCheckboxes].filter(cb => cb.checked).length;
        deleteSelectedBtn.style.display = checkedCount > 0 ? 'inline-block' : 'none';
        document.getElementById('selectedCount').textContent = checkedCount;
    }

    // Handle delete selected button
    deleteSelectedBtn.addEventListener('click', function() {
        const selectedStudents = [...studentCheckboxes].filter(cb => cb.checked);
        const studentIds = selectedStudents.map(cb => cb.value);
        const userIds = selectedStudents.map(cb => cb.dataset.userId);
        
        document.getElementById('bulk_delete_student_ids').value = studentIds.join(',');
        document.getElementById('bulk_delete_user_ids').value = userIds.join(',');
        document.getElementById('delete_count').textContent = selectedStudents.length;
        
        bulkDeleteModal.show();
    });

    // Edit student
    const editButtons = document.querySelectorAll('.edit-student');
    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            const data = this.dataset;
            
            // Populate the form fields
            document.getElementById('edit_student_id').value = data.id || '';
            document.getElementById('edit_student_number').value = data.studentNumber || '';
            document.getElementById('edit_first_name').value = data.firstName || '';
            document.getElementById('edit_last_name').value = data.lastName || '';
            document.getElementById('edit_email').value = data.email || (data.studentNumber ? data.studentNumber + '@example.com' : '');
            document.getElementById('edit_contact').value = data.contact || '';
            document.getElementById('edit_address').value = data.address || '';
            document.getElementById('edit_program').value = data.program || '';
            document.getElementById('edit_year_level').value = data.year || '1';
            document.getElementById('edit_academic_year').value = data.academicYear || '';
            
            const editModal = new bootstrap.Modal(document.getElementById('editStudentModal'));
            editModal.show();
        });
    });

    // Delete student
    const deleteButtons = document.querySelectorAll('.delete-student');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
            const data = this.dataset;
            document.getElementById('delete_student_id').value = data.id;
            document.getElementById('delete_user_id').value = data.userId;
            document.getElementById('delete_student_name').textContent = data.name;
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteStudentModal'));
            deleteModal.show();
        });
    });
});
</script>

<?php 
// Add bulk delete JavaScript
?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize checkboxes and button
    const selectAll = document.getElementById('selectAll');
    const studentCheckboxes = document.querySelectorAll('.student-checkbox');
    const deleteSelectedBtn = document.getElementById('deleteSelectedBtn');
    const selectedCountSpan = document.getElementById('selectedCount');
    
    // Function to update delete button state
    function updateDeleteButton() {
        const checkedCount = Array.from(studentCheckboxes).filter(cb => cb.checked).length;
        deleteSelectedBtn.style.display = checkedCount > 0 ? 'inline-block' : 'none';
        selectedCountSpan.textContent = checkedCount;
    }

    // Handle select all checkbox
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            studentCheckboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
            updateDeleteButton();
        });
    }

    // Handle individual checkboxes
    studentCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const allChecked = Array.from(studentCheckboxes).every(cb => cb.checked);
            if (selectAll) {
                selectAll.checked = allChecked;
            }
            updateDeleteButton();
        });
    });

    // Handle delete selected button
    if (deleteSelectedBtn) {
        deleteSelectedBtn.addEventListener('click', function() {
            const selectedStudents = Array.from(studentCheckboxes).filter(cb => cb.checked);
            const studentIds = selectedStudents.map(cb => cb.value);
            const userIds = selectedStudents.map(cb => cb.dataset.userId);
            
            document.getElementById('bulk_delete_student_ids').value = studentIds.join(',');
            document.getElementById('bulk_delete_user_ids').value = userIds.join(',');
            document.getElementById('delete_count').textContent = selectedStudents.length;
            
            const bulkDeleteModal = new bootstrap.Modal(document.getElementById('bulkDeleteModal'));
            bulkDeleteModal.show();
        });
    }
});
</script>
<?php
require_once '../includes/footer.php';
?>