<?php

class StudentController {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->conn = $this->db->getConnection();
    }

    public function getStudents($page = 1, $perPage = 10) {
        try {
            $offset = ($page - 1) * $perPage;
            
            // Get total count
            $count_stmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM students s 
                JOIN users u ON s.user_id = u.id
            ");
            $count_stmt->execute();
            $total = $count_stmt->get_result()->fetch_assoc()['total'];
            
            // Get students for current page
            $stmt = $this->conn->prepare("
                SELECT 
                    s.*, 
                    u.username,
                    u.email,
                    u.id as user_id,
                    p.program_code,
                    p.program_name,
                    ay.year_name as academic_year
                FROM students s
                JOIN users u ON s.user_id = u.id
                LEFT JOIN programs p ON s.program_id = p.id
                LEFT JOIN academic_years ay ON p.academic_year_id = ay.id
                ORDER BY s.student_number
                LIMIT ? OFFSET ?
            ");
            $stmt->bind_param("ii", $perPage, $offset);
            $stmt->execute();
            
            return [
                'students' => $stmt->get_result()->fetch_all(MYSQLI_ASSOC),
                'total' => $total,
                'pages' => ceil($total / $perPage)
            ];
        } catch (Exception $e) {
            throw new Exception("Error fetching students: " . $e->getMessage());
        }
    }

    public function resetPassword($userId, $studentNumber) {
        try {
            $this->db->beginTransaction();
            
            // Verify student exists
            $stmt = $this->conn->prepare("
                SELECT id 
                FROM users 
                WHERE id = ? AND role = 'student'
            ");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            
            if ($stmt->get_result()->num_rows === 0) {
                throw new Exception("Student not found");
            }
            
            // Reset password
            $password_hash = password_hash($studentNumber, PASSWORD_DEFAULT);
            $stmt = $this->conn->prepare("
                UPDATE users 
                SET password = ? 
                WHERE id = ?
            ");
            $stmt->bind_param("si", $password_hash, $userId);
            
            if (!$stmt->execute()) {
                throw new Exception("Error resetting password");
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            throw new Exception("Password reset failed: " . $e->getMessage());
        }
    }

    public function transferStudent($studentId, $newProgramId, $newYearLevel) {
        try {
            $this->db->beginTransaction();
            
            // Verify student exists
            $stmt = $this->conn->prepare("
                SELECT id 
                FROM students 
                WHERE id = ?
            ");
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            
            if ($stmt->get_result()->num_rows === 0) {
                throw new Exception("Student not found");
            }
            
            // Update student program
            $stmt = $this->conn->prepare("
                UPDATE students 
                SET program_id = ?, year_level = ? 
                WHERE id = ?
            ");
            $stmt->bind_param("iii", $newProgramId, $newYearLevel, $studentId);
            
            if (!$stmt->execute()) {
                throw new Exception("Error transferring student");
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            throw new Exception("Transfer failed: " . $e->getMessage());
        }
    }

    public function deleteStudent($studentId, $userId) {
        try {
            $this->db->beginTransaction();
            
            // Delete payments
            $stmt = $this->conn->prepare("
                DELETE p 
                FROM payments p 
                INNER JOIN student_fees sf ON p.student_fee_id = sf.id 
                WHERE sf.student_id = ?
            ");
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            
            // Delete student fees
            $stmt = $this->conn->prepare("
                DELETE FROM student_fees 
                WHERE student_id = ?
            ");
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            
            // Delete student record
            $stmt = $this->conn->prepare("
                DELETE FROM students 
                WHERE id = ?
            ");
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            
            // Delete user account
            $stmt = $this->conn->prepare("
                DELETE FROM users 
                WHERE id = ? AND role = 'student'
            ");
            $stmt->bind_param("i", $userId);
            
            if (!$stmt->execute()) {
                throw new Exception("Error deleting user account");
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            throw new Exception("Delete failed: " . $e->getMessage());
        }
    }

    public function getActivePrograms() {
        try {
            $result = $this->conn->query("
                SELECT id, program_code, program_name 
                FROM programs 
                WHERE status = 'active' 
                ORDER BY program_code
            ");
            
            return $result->fetch_all(MYSQLI_ASSOC);
        } catch (Exception $e) {
            throw new Exception("Error fetching programs: " . $e->getMessage());
        }
    }
}