<?php
class Database {
    private $conn = null;
    private static $instance = null;
    private $inTransaction = false;

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        require_once __DIR__ . '/../config/config.php';
        
        try {
            $this->conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
            
            if ($this->conn->connect_error) {
                throw new Exception("Connection failed: " . $this->conn->connect_error);
            }
            
            // Set character set to utf8mb4
            if (!$this->conn->set_charset("utf8mb4")) {
                throw new Exception("Error setting charset utf8mb4: " . $this->conn->error);
            }
            
            // Set strict mode for better error handling
            $this->conn->query("SET SESSION sql_mode = 'STRICT_ALL_TABLES'");
        } catch (Exception $e) {
            error_log("Database connection error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get singleton instance
     */
    public static function getInstance(): Database {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Get the database connection
     */
    public function getConnection(): mysqli {
        return $this->conn;
    }

    /**
     * Begin a transaction with proper error handling
     */
    public function beginTransaction(): bool {
        if ($this->inTransaction) {
            throw new Exception("Transaction already in progress");
        }
        
        if (!$this->conn->begin_transaction()) {
            throw new Exception("Failed to start transaction: " . $this->conn->error);
        }
        
        $this->inTransaction = true;
        return true;
    }

    /**
     * Commit a transaction with proper error handling
     */
    public function commit(): bool {
        if (!$this->inTransaction) {
            throw new Exception("No transaction in progress");
        }
        
        if (!$this->conn->commit()) {
            throw new Exception("Failed to commit transaction: " . $this->conn->error);
        }
        
        $this->inTransaction = false;
        return true;
    }

    /**
     * Rollback a transaction with proper error handling
     */
    public function rollback(): bool {
        if (!$this->inTransaction) {
            throw new Exception("No transaction in progress");
        }
        
        if (!$this->conn->rollback()) {
            throw new Exception("Failed to rollback transaction: " . $this->conn->error);
        }
        
        $this->inTransaction = false;
        return true;
    }

    /**
     * Prepare a statement with proper error handling
     */
    public function prepare(string $query): mysqli_stmt {
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            throw new Exception("Prepare failed: " . $this->conn->error);
        }
        return $stmt;
    }

    /**
     * Execute a query with proper error handling
     */
    public function query(string $query) {
        $result = $this->conn->query($query);
        if ($result === false) {
            throw new Exception("Query failed: " . $this->conn->error);
        }
        return $result;
    }

    /**
     * Close the database connection properly
     */
    public function close(): void {
        if ($this->inTransaction) {
            $this->rollback();
        }
        if ($this->conn !== null) {
            $this->conn->close();
            $this->conn = null;
        }
    }

    /**
     * Clean up resources in destructor
     */
    public function __destruct() {
        $this->close();
    }

    /**
     * Prevent cloning of the instance
     */
    private function __clone() { }

    /**
     * Prevent unserializing of the instance
     */
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}