<?php
function calculateDueDate($conn, $academic_year_id, $fee_type_id) {
    // Get academic year details
    $stmt = $conn->prepare("
        SELECT 
            start_date,
            end_date,
            semester 
        FROM academic_years 
        WHERE id = ?
    ");
    $stmt->bind_param("i", $academic_year_id);
    $stmt->execute();
    $academic_year = $stmt->get_result()->fetch_assoc();

    // Get fee type payment schedule
    $stmt = $conn->prepare("
        SELECT 
            payment_schedule,
            grace_period_days 
        FROM fee_types 
        WHERE id = ?
    ");
    $stmt->bind_param("i", $fee_type_id);
    $stmt->execute();
    $fee_type = $stmt->get_result()->fetch_assoc();

    $due_date = null;
    
    // Default grace period if not set
    $grace_period = $fee_type['grace_period_days'] ?? 30;

    switch($fee_type['payment_schedule']) {
        case 'start_of_semester':
            $due_date = date('Y-m-d', strtotime($academic_year['start_date'] . " +{$grace_period} days"));
            break;
        case 'mid_semester':
            $semester_duration = ceil((strtotime($academic_year['end_date']) - strtotime($academic_year['start_date'])) / 2);
            $mid_semester = date('Y-m-d', strtotime($academic_year['start_date'] . " +{$semester_duration} seconds"));
            $due_date = date('Y-m-d', strtotime($mid_semester . " +{$grace_period} days"));
            break;
        case 'end_of_semester':
            $due_date = date('Y-m-d', strtotime($academic_year['end_date'] . " -{$grace_period} days"));
            break;
        default:
            // Default to 30 days from current date for immediate fees
            $due_date = date('Y-m-d', strtotime("+{$grace_period} days"));
    }

    return $due_date;
}

function updateDueDates($conn) {
    // Get current academic year
    $current_year = $conn->query("SELECT id FROM academic_years WHERE is_current = 1")->fetch_assoc();
    
    if (!$current_year) {
        return false;
    }

    // Get all fees that need due date updates
    $stmt = $conn->prepare("
        SELECT 
            sf.id,
            sf.fee_type_id,
            sf.academic_year_id,
            sf.status,
            ft.payment_schedule
        FROM student_fees sf
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        WHERE sf.status = 'pending'
        AND sf.academic_year_id = ?
    ");
    $stmt->bind_param("i", $current_year['id']);
    $stmt->execute();
    $fees = $stmt->get_result();

    // Update statement
    $update_stmt = $conn->prepare("
        UPDATE student_fees 
        SET due_date = ?, 
            last_updated = NOW() 
        WHERE id = ?
    ");

    while ($fee = $fees->fetch_assoc()) {
        $new_due_date = calculateDueDate($conn, $fee['academic_year_id'], $fee['fee_type_id']);
        $update_stmt->bind_param("si", $new_due_date, $fee['id']);
        $update_stmt->execute();
    }

    return true;
}

function getDueStatus($due_date) {
    $today = strtotime('today');
    $due = strtotime($due_date);
    $diff = $due - $today;
    $days = ceil($diff / (60 * 60 * 24));

    if ($days < 0) {
        return [
            'status' => 'overdue',
            'days' => abs($days),
            'class' => 'text-danger'
        ];
    } elseif ($days == 0) {
        return [
            'status' => 'due today',
            'days' => 0,
            'class' => 'text-warning'
        ];
    } else {
        return [
            'status' => 'upcoming',
            'days' => $days,
            'class' => 'text-success'
        ];
    }
}