<?php
function assignFeesToProgram($conn, $fee_type_id, $program_id) {
    // Get all students in the program
    $stmt = $conn->prepare("
        SELECT id 
        FROM students 
        WHERE program_id = ?
    ");
    $stmt->bind_param("i", $program_id);
    $stmt->execute();
    $students = $stmt->get_result();

    // Get the current academic year
    $academic_year = $conn->query("
        SELECT id 
        FROM academic_years 
        WHERE is_current = 1
    ")->fetch_assoc();

    if (!$academic_year) {
        throw new Exception("No current academic year found");
    }

    // Get fee type details
    $stmt = $conn->prepare("
        SELECT amount, is_recurring 
        FROM fee_types 
        WHERE id = ?
    ");
    $stmt->bind_param("i", $fee_type_id);
    $stmt->execute();
    $fee_type = $stmt->get_result()->fetch_assoc();

    if (!$fee_type) {
        throw new Exception("Fee type not found");
    }

    // Set due date to 30 days from now
    $due_date = date('Y-m-d', strtotime('+30 days'));

    // Prepare statement for inserting student fees
    $stmt = $conn->prepare("
        INSERT INTO student_fees (
            student_id, 
            fee_type_id, 
            academic_year_id, 
            amount, 
            due_date, 
            status
        ) VALUES (?, ?, ?, ?, ?, 'pending')
    ");

    // Assign fees to each student
    while ($student = $students->fetch_assoc()) {
        $stmt->bind_param(
            "iiids", 
            $student['id'], 
            $fee_type_id, 
            $academic_year['id'], 
            $fee_type['amount'], 
            $due_date
        );
        $stmt->execute();
    }
}

function removeFeesFromProgram($conn, $fee_type_id, $program_id) {
    // Get current academic year
    $academic_year = $conn->query("
        SELECT id 
        FROM academic_years 
        WHERE is_current = 1
    ")->fetch_assoc();

    if (!$academic_year) {
        throw new Exception("No current academic year found");
    }

    // Remove fees for students in the program that haven't been paid
    $stmt = $conn->prepare("
        DELETE sf FROM student_fees sf
        JOIN students s ON sf.student_id = s.id
        WHERE sf.fee_type_id = ?
        AND s.program_id = ?
        AND sf.academic_year_id = ?
        AND sf.status = 'pending'
    ");
    $stmt->bind_param("iii", $fee_type_id, $program_id, $academic_year['id']);
    $stmt->execute();
}

function getApplicableFeesForStudent($conn, $student_id) {
    // Get student's program
    $stmt = $conn->prepare("
        SELECT program_id 
        FROM students 
        WHERE id = ?
    ");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();

    if (!$student) {
        throw new Exception("Student not found");
    }

    // Get applicable fees
    $stmt = $conn->prepare("
        SELECT DISTINCT ft.*
        FROM fee_types ft
        LEFT JOIN fee_type_programs ftp ON ft.id = ftp.fee_type_id
        WHERE ft.status = 'active'
        AND (
            ft.applicable_to = 'all'
            OR (
                ft.applicable_to = 'specific_programs'
                AND ftp.program_id = ?
            )
        )
        ORDER BY ft.name
    ");
    $stmt->bind_param("i", $student['program_id']);
    $stmt->execute();
    return $stmt->get_result();
}
?>