<?php
function calculateStudentFinancials($conn, $student_id, $year_level = null) {
    $financials = [
        'total_fees' => 0,
        'total_payments' => 0,
        'total_refunds' => 0,
        'provisional_fees' => 0,
        'year_payments' => 0,
        'existing_balance' => 0,
        'balance_forward' => 0,
        'pending_refunds' => 0,
        'current_year_payments' => 0,
        'overall_balance' => 0
    ];

    // Get current year
    $current_year = date('Y');

    // Get current academic year
    $current_academic_year = $conn->query("SELECT id FROM academic_years WHERE is_current = 1")->fetch_assoc();

    if ($year_level === null) {
        // Get student's year level if not provided
        $stmt = $conn->prepare("SELECT year_level FROM students WHERE id = ?");
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $year_level = $result['year_level'];
    }

    // Total Fees Query
    $stmt = $conn->prepare("
        SELECT COALESCE(SUM(sf.amount), 0) as amount 
        FROM student_fees sf 
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        WHERE sf.student_id = ? 
        AND ft.level = CONCAT('Level ', CASE ?
            WHEN 1 THEN '100'
            WHEN 2 THEN '200'
            WHEN 3 THEN '300'
            WHEN 4 THEN '400'
            ELSE '100'
        END)
    ");
    $stmt->bind_param("ii", $student_id, $year_level);
    $stmt->execute();
    $financials['total_fees'] = $stmt->get_result()->fetch_assoc()['amount'] ?? 0;

    // Total Payments and Refunds Query
    $stmt = $conn->prepare("
        SELECT 
            COALESCE(SUM(CASE WHEN p.status = 'confirmed' THEN p.amount ELSE 0 END), 0) as amount,
            COALESCE(SUM(CASE WHEN p.status = 'refunded' THEN COALESCE(p.refunded_amount, p.amount) ELSE 0 END), 0) as refunded_amount
        FROM payments p 
        JOIN student_fees sf ON p.student_fee_id = sf.id 
        WHERE sf.student_id = ?
    ");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $financials['total_payments'] = ($result['amount'] ?? 0);
    $financials['total_payments_refunded'] = ($result['refunded_amount'] ?? 0);
    $financials['total_payments'] = $financials['total_payments'] - $financials['total_payments_refunded'];

    // Provisional Fees (Current Year) Query
    $stmt = $conn->prepare("
        SELECT COALESCE(SUM(sf.amount), 0) as amount 
        FROM student_fees sf 
        JOIN academic_years ay ON sf.academic_year_id = ay.id 
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        WHERE sf.student_id = ? 
        AND ay.is_current = 1 
        AND ft.level = CONCAT('Level ', CASE ?
            WHEN 1 THEN '100'
            WHEN 2 THEN '200'
            WHEN 3 THEN '300'
            WHEN 4 THEN '400'
            ELSE '100'
        END)
    ");
    $stmt->bind_param("ii", $student_id, $year_level);
    $stmt->execute();
    $financials['provisional_fees'] = $stmt->get_result()->fetch_assoc()['amount'] ?? 0;

    // Current Year and Academic Year Payments Query
    $stmt = $conn->prepare("
        SELECT 
            -- Calendar year payments
            COALESCE(SUM(CASE 
                WHEN YEAR(p.payment_date) = ? AND p.status = 'confirmed' 
                THEN p.amount ELSE 0 
            END), 0) as year_amount,
            COALESCE(SUM(CASE 
                WHEN YEAR(p.payment_date) = ? AND p.status = 'refunded' 
                THEN COALESCE(p.refunded_amount, p.amount) 
                ELSE 0 
            END), 0) as year_refunded_amount,
            
            -- Current academic year payments
            COALESCE(SUM(CASE 
                WHEN ay.is_current = 1 AND p.status = 'confirmed' 
                THEN p.amount ELSE 0 
            END), 0) as academic_year_amount,
            COALESCE(SUM(CASE 
                WHEN ay.is_current = 1 AND p.status = 'refunded' 
                THEN COALESCE(p.refunded_amount, p.amount) 
                ELSE 0 
            END), 0) as academic_year_refunded_amount
        FROM payments p 
        JOIN student_fees sf ON p.student_fee_id = sf.id 
        JOIN academic_years ay ON sf.academic_year_id = ay.id
        WHERE sf.student_id = ?
    ");
    $stmt->bind_param("iii", $current_year, $current_year, $student_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();

    // Calendar year payments
    $financials['year_payments'] = ($result['year_amount'] ?? 0);
    $financials['year_payments_refunded'] = ($result['year_refunded_amount'] ?? 0);
    $financials['year_payments'] = $financials['year_payments'] - $financials['year_payments_refunded'];

    // Academic year payments
    $financials['current_year_payments'] = ($result['academic_year_amount'] ?? 0);
    $financials['current_year_payments_refunded'] = ($result['academic_year_refunded_amount'] ?? 0);
    $financials['current_year_payments'] = $financials['current_year_payments'] - $financials['current_year_payments_refunded'];

    // Pending Refunds Query
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count, COALESCE(SUM(p.amount), 0) as amount
        FROM payments p 
        JOIN student_fees sf ON p.student_fee_id = sf.id 
        WHERE sf.student_id = ? AND p.status = 'refund_pending'
    ");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $financials['pending_refunds_count'] = $result['count'] ?? 0;
    $financials['pending_refunds_amount'] = $result['amount'] ?? 0;

    // Calculate balances
    $financials['existing_balance'] = $financials['total_fees'] - $financials['total_payments'];
    $financials['balance_forward'] = $financials['existing_balance'] + $financials['provisional_fees'];

    return $financials;
}