<?php
// Format amount with Ghana Cedis symbol
function formatMoney($amount) {
    return 'GH₵ ' . number_format($amount, 2);
}

// Format amount with Ghana Cedis symbol (alias for consistency)
function formatPeso($amount) {
    return formatMoney($amount); // Use the same Ghana Cedis formatting
}

// Generate a random student number
function generateStudentNumber($year) {
    $year_code = substr($year, 2, 2);
    $random = mt_rand(1000, 9999);
    return $year_code . '-' . $random;
}

// Validate date format
function isValidDate($date, $format = 'Y-m-d') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

// Clean input data
function cleanInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    return htmlspecialchars($data);
}

// Get current academic year
function getCurrentAcademicYear($conn) {
    $stmt = $conn->prepare("SELECT year_name FROM academic_years WHERE is_current = 1 LIMIT 1");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['year_name'];
    }
    return date('Y') . '-' . (date('Y') + 1);
}

// Check if student has pending payments
function hasUnpaidFees($conn, $student_id) {
    $stmt = $conn->prepare("
        SELECT COUNT(*) as unpaid_count 
        FROM student_fees 
        WHERE student_id = ? AND status = 'unpaid'
    ");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['unpaid_count'] > 0;
}

// Get student balance
function getStudentBalance($conn, $student_id) {
    $stmt = $conn->prepare("
        SELECT 
            SUM(sf.amount) as total_fees,
            (SELECT COALESCE(SUM(amount), 0) 
             FROM payments p 
             JOIN student_fees sf2 ON p.student_fee_id = sf2.id 
             WHERE sf2.student_id = ? AND p.status = 'confirmed') as total_paid
        FROM student_fees sf 
        WHERE sf.student_id = ?
    ");
    $stmt->bind_param("ii", $student_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['total_fees'] - $row['total_paid'];
}

// Send email notification
function sendEmailNotification($to, $subject, $message) {
    $headers = "MIME-Version: 1.0" . "\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $headers .= 'From: KCOE Fee Payment System <noreply@kcoe.edu>' . "\r\n";
    
    return mail($to, $subject, $message, $headers);
}

// Generate payment reference number
function generateReferenceNumber() {
    $prefix = 'PAY';
    $timestamp = time();
    $random = mt_rand(1000, 9999);
    return $prefix . $timestamp . $random;
}

// Check if payment is still within grace period
function isWithinGracePeriod($due_date, $grace_days = 7) {
    $due = new DateTime($due_date);
    $now = new DateTime();
    $diff = $now->diff($due);
    return $diff->days <= $grace_days;
}