<?php
function getPaymentsList($conn, $search = '', $status = '', $date_from = '', $date_to = '', $page = 1, $per_page = 10) {
    $offset = ($page - 1) * $per_page;
    
    // Build query conditions
    $conditions = [];
    $params = [];
    $types = "";

    if ($status && $status !== 'all') {
        $conditions[] = "p.status = ?";
        $params[] = $status;
        $types .= "s";
    }

    if ($date_from) {
        $conditions[] = "DATE(p.created_at) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }

    if ($date_to) {
        $conditions[] = "DATE(p.created_at) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }

    if ($search) {
        $conditions[] = "(s.student_number LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ? OR p.reference_number LIKE ?)";
        $search_param = "%$search%";
        $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
        $types .= "ssss";
    }

    // Construct WHERE clause
    $where_clause = !empty($conditions) ? " WHERE " . implode(" AND ", $conditions) : "";

    // Get total count for pagination
    $count_query = "
        SELECT COUNT(*) as total
        FROM payments p
        JOIN student_fees sf ON p.student_fee_id = sf.id
        JOIN students s ON sf.student_id = s.id
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        $where_clause
    ";

    $stmt = $conn->prepare($count_query);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $total_records = $stmt->get_result()->fetch_assoc()['total'];
    $total_pages = ceil($total_records / $per_page);

    // Get paginated results
    $query = "
        SELECT 
            p.*,
            s.student_number,
            s.first_name,
            s.last_name,
            ft.name as fee_type,
            CASE 
                WHEN p.status = 'pending' THEN p.created_at
                ELSE COALESCE(p.payment_date, p.created_at)
            END as effective_date
        FROM payments p
        JOIN student_fees sf ON p.student_fee_id = sf.id
        JOIN students s ON sf.student_id = s.id
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        $where_clause
        ORDER BY effective_date DESC
        LIMIT ? OFFSET ?
    ";

    // Add pagination parameters
    $params[] = $per_page;
    $params[] = $offset;
    $types .= "ii";

    $stmt = $conn->prepare($query);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    
    return [
        'results' => $stmt->get_result(),
        'total_pages' => $total_pages,
        'current_page' => $page,
        'total_records' => $total_records
    ];
}

function renderPaymentsList($payments_data, $search = '', $status = '', $date_from = '', $date_to = '') {
    $payments = $payments_data['results'];
    ob_start();
    ?>
    <!-- Filter Form -->
    <form method="GET" class="row g-3 mb-3">
        <div class="col-md-3">
            <input type="text" name="search" class="form-control form-control-sm" 
                   placeholder="Search..." value="<?php echo htmlspecialchars($search); ?>">
        </div>
        <div class="col-md-2">
            <select name="status" class="form-select form-select-sm">
                <option value="">All Status</option>
                <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                <option value="confirmed" <?php echo $status === 'confirmed' ? 'selected' : ''; ?>>Confirmed</option>
                <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
            </select>
        </div>
        <div class="col-md-2">
            <input type="date" name="date_from" class="form-control form-control-sm" 
                   value="<?php echo htmlspecialchars($date_from); ?>">
        </div>
        <div class="col-md-2">
            <input type="date" name="date_to" class="form-control form-control-sm" 
                   value="<?php echo htmlspecialchars($date_to); ?>">
        </div>
        <div class="col-md-3">
            <button type="submit" class="btn btn-primary btn-sm">Filter</button>
            <a href="?page=1" class="btn btn-secondary btn-sm">Reset</a>
        </div>
    </form>

    <!-- Payments Table -->
    <div class="table-responsive p-0">
        <table class="table align-items-center mb-0">
            <thead>
                <tr>
                    <th>Student</th>
                    <th>Payment Details</th>
                    <th>Status</th>
                    <th>Date</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($payment = $payments->fetch_assoc()): ?>
                    <tr>
                        <td>
                            <div class="d-flex align-items-center">
                                <div>
                                    <h6 class="mb-0 text-sm"><?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?></h6>
                                    <p class="text-xs text-secondary mb-0"><?php echo htmlspecialchars($payment['student_number']); ?></p>
                                </div>
                            </div>
                        </td>
                        <td>
                            <p class="text-sm font-weight-bold mb-0"><?php echo formatMoney($payment['amount']); ?></p>
                            <p class="text-xs text-secondary mb-0"><?php echo htmlspecialchars($payment['fee_type']); ?></p>
                        </td>
                        <td>
                            <span class="badge badge-sm bg-<?php 
                                echo $payment['status'] === 'confirmed' ? 'success' : 
                                    ($payment['status'] === 'pending' ? 'warning' : 'danger'); 
                            ?>">
                                <?php echo ucfirst($payment['status']); ?>
                            </span>
                        </td>
                        <td>
                            <p class="text-sm font-weight-bold mb-0">
                                <?php echo formatDateTime($payment['effective_date']); ?>
                            </p>
                            <p class="text-xs text-secondary mb-0">
                                <?php 
                                if ($payment['status'] === 'pending') {
                                    echo 'Created: ' . formatDateTime($payment['created_at']);
                                } else {
                                    echo 'Paid: ' . formatDateTime($payment['payment_date']);
                                }
                                ?>
                            </p>
                        </td>
                    </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($payments_data['total_pages'] > 1): ?>
    <div class="d-flex justify-content-center mt-4">
        <nav>
            <ul class="pagination pagination-sm">
                <?php for ($i = 1; $i <= $payments_data['total_pages']; $i++): ?>
                    <li class="page-item <?php echo $i === $payments_data['current_page'] ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>&date_from=<?php echo urlencode($date_from); ?>&date_to=<?php echo urlencode($date_to); ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                <?php endfor; ?>
            </ul>
        </nav>
    </div>
    <?php endif; ?>
    <?php
    return ob_get_clean();
}
