<?php
require_once '../config/config.php';
require_once '../includes/functions.php';

// Function to create refunds table if it doesn't exist
function create_refunds_table($conn) {
    $sql = "CREATE TABLE IF NOT EXISTS payment_refunds (
        id INT PRIMARY KEY AUTO_INCREMENT,
        payment_id INT NOT NULL,
        refund_reference VARCHAR(100) NOT NULL,
        amount DECIMAL(10,2) NOT NULL,
        status ENUM('pending', 'completed', 'failed') NOT NULL DEFAULT 'pending',
        reason TEXT,
        refunded_by INT,
        refunded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        payment_data TEXT,
        FOREIGN KEY (payment_id) REFERENCES payments(id),
        FOREIGN KEY (refunded_by) REFERENCES users(id)
    )";
    
    if ($conn->query($sql) === FALSE) {
        die("Error creating refunds table: " . $conn->error);
    }
}

// Function to process a refund
function process_paystack_refund($payment_id, $amount, $reason, $refunded_by) {
    global $conn;
    
    // Get payment details
    $stmt = $conn->prepare("
        SELECT p.*, sf.student_id, sf.academic_year_id, sf.fee_type_id
        FROM payments p
        JOIN student_fees sf ON p.student_fee_id = sf.id
        WHERE p.id = ?
    ");
    $stmt->bind_param("i", $payment_id);
    $stmt->execute();
    $payment = $stmt->get_result()->fetch_assoc();
    
    if (!$payment) {
        return [
            'success' => false,
            'message' => 'Payment not found'
        ];
    }
    
    // Initialize Paystack refund
    $url = "https://api.paystack.co/refund";
    $fields = [
        'transaction' => $payment['reference_number'],
        'amount' => $amount * 100, // Convert to kobo
        'reason' => $reason
    ];
    
    $headers = [
        'Authorization: Bearer ' . PAYSTACK_SECRET_KEY,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    
    if ($err) {
        return [
            'success' => false,
            'message' => 'Curl Error: ' . $err
        ];
    }
    
    $result = json_decode($response, true);
    
    // Record the refund in our database
    $stmt = $conn->prepare("
        INSERT INTO payment_refunds (
            payment_id, refund_reference, amount, status,
            reason, refunded_by, payment_data
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $status = $result['status'] ? 'completed' : 'failed';
    $refund_ref = $result['data']['reference'] ?? null;
    $payment_data = json_encode($result);
    
    $stmt->bind_param("isdsiss",
        $payment_id,
        $refund_ref,
        $amount,
        $status,
        $reason,
        $refunded_by,
        $payment_data
    );
    
    if ($stmt->execute()) {
        // Update the original payment
        $stmt = $conn->prepare("
            UPDATE payments 
            SET 
                status = 'refunded',
                refunded_amount = refunded_amount + ?,
                updated_at = CURRENT_TIMESTAMP,
                payment_data = JSON_SET(
                    COALESCE(payment_data, '{}'),
                    '$.refund_reference', ?,
                    '$.refund_status', ?,
                    '$.refund_date', CURRENT_TIMESTAMP
                )
            WHERE id = ?
        ");
        
        $stmt->bind_param("dssi",
            $amount,
            $refund_ref,
            $status,
            $payment_id
        );
        
        $stmt->execute();
        
        return [
            'success' => true,
            'message' => 'Refund processed successfully',
            'data' => $result['data']
        ];
    } else {
        return [
            'success' => false,
            'message' => 'Failed to record refund'
        ];
    }
}

// Function to check refund status
function check_refund_status($refund_reference) {
    $url = "https://api.paystack.co/refund/" . urlencode($refund_reference);
    $headers = [
        'Authorization: Bearer ' . PAYSTACK_SECRET_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    
    if ($err) {
        return [
            'success' => false,
            'message' => 'Curl Error: ' . $err
        ];
    }
    
    return json_decode($response, true);
}
?>