<?php
function getStudentProgramFees($conn, $student_id, $academic_year_id = null) {
    $query = "
        SELECT 
            ft.*,
            ay.year_name,
            sf.id as student_fee_id,
            sf.status as payment_status,
            sf.due_date,
            COALESCE(
                (SELECT SUM(amount) 
                 FROM payments 
                 WHERE student_fee_id = sf.id 
                 AND status = 'confirmed'), 
                0
            ) as amount_paid
        FROM students s
        JOIN program_fees pf ON s.program_id = pf.program_id
        JOIN fee_types ft ON pf.fee_type_id = ft.id
        JOIN academic_years ay ON pf.academic_year_id = ay.id
        LEFT JOIN student_fees sf ON (
            sf.student_id = s.id 
            AND sf.fee_type_id = ft.id 
            AND sf.academic_year_id = ay.id
        )
        WHERE s.id = ?
    ";
    
    $params = [$student_id];
    $types = "i";
    
    if ($academic_year_id) {
        $query .= " AND ay.id = ?";
        $params[] = $academic_year_id;
        $types .= "i";
    }
    
    $query .= " ORDER BY ay.year_name DESC, ft.name";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    
    return $stmt->get_result();
}

function assignFeesToNewStudent($conn, $student_id, $program_id) {
    // Get current academic year
    $current_academic_year = $conn->query("
        SELECT id FROM academic_years 
        WHERE is_current = 1 AND status = 'active'
        LIMIT 1
    ")->fetch_assoc();

    if (!$current_academic_year) {
        throw new Exception("No active academic year found");
    }

    // Get all fees for the program
    $stmt = $conn->prepare("
        SELECT 
            pf.program_id,
            pf.fee_type_id,
            ft.amount,
            ft.due_date
        FROM program_fees pf
        JOIN fee_types ft ON pf.fee_type_id = ft.id
        WHERE pf.program_id = ?
        AND pf.academic_year_id = ?
    ");

    $stmt->bind_param("ii", $program_id, $current_academic_year['id']);
    $stmt->execute();
    $program_fees = $stmt->get_result();

    // Begin transaction
    $conn->begin_transaction();

    try {
        // Insert student fees for each program fee
        $insert_stmt = $conn->prepare("
            INSERT INTO student_fees (
                student_id,
                fee_type_id,
                academic_year_id,
                amount,
                due_date,
                status
            ) VALUES (?, ?, ?, ?, ?, 'pending')
        ");

        while ($fee = $program_fees->fetch_assoc()) {
            $insert_stmt->bind_param(
                "iiids",
                $student_id,
                $fee['fee_type_id'],
                $current_academic_year['id'],
                $fee['amount'],
                $fee['due_date']
            );
            $insert_stmt->execute();
        }

        $conn->commit();
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }

    return true;
}

function get_program_fees($conn, $program_id) {
    // Get current academic year
    $academic_year = $conn->query("
        SELECT * 
        FROM academic_years 
        WHERE is_current = 1
    ")->fetch_assoc();
    
    if (!$academic_year) {
        throw new Exception("No active academic year found");
    }
    
    // Get program fees
    $stmt = $conn->prepare("
        SELECT ft.id as fee_type_id, ft.amount
        FROM program_fees pf
        JOIN fee_types ft ON pf.fee_type_id = ft.id
        WHERE pf.program_id = ?
        AND pf.academic_year_id = ?
    ");
    $stmt->bind_param("ii", $program_id, $academic_year['id']);
    $stmt->execute();
    $program_fees = $stmt->get_result();
    
    // Insert student fees
    $stmt = $conn->prepare("
        INSERT INTO student_fees (
            student_id, 
            fee_type_id, 
            academic_year_id, 
            amount, 
            due_date, 
            status
        ) VALUES (?, ?, ?, ?, DATE_ADD(CURRENT_DATE, INTERVAL 30 DAY), 'pending')
    ");
    
    while ($fee = $program_fees->fetch_assoc()) {
        $stmt->bind_param(
            "iiid", 
            $student_id, 
            $fee['fee_type_id'], 
            $academic_year['id'], 
            $fee['amount']
        );
        $stmt->execute();
    }
}

function updateStudentFees($conn, $student_id, $program_id) {
    $conn->begin_transaction();
    
    try {
        // Get current academic year
        $academic_year = $conn->query("
            SELECT id 
            FROM academic_years 
            WHERE is_current = 1
        ")->fetch_assoc();
        
        if (!$academic_year) {
            throw new Exception("No active academic year found");
        }
        
        // Remove pending fees that are no longer applicable
        $stmt = $conn->prepare("
            DELETE sf FROM student_fees sf
            WHERE sf.student_id = ?
            AND sf.academic_year_id = ?
            AND sf.status = 'pending'
            AND NOT EXISTS (
                SELECT 1 
                FROM program_fees pf 
                WHERE pf.program_id = ?
                AND pf.fee_type_id = sf.fee_type_id
                AND pf.academic_year_id = sf.academic_year_id
            )
        ");
        $stmt->bind_param("iii", $student_id, $academic_year['id'], $program_id);
        $stmt->execute();
        
        // Add new applicable fees
        $stmt = $conn->prepare("
            INSERT INTO student_fees (
                student_id, 
                fee_type_id, 
                academic_year_id, 
                amount, 
                due_date, 
                status
            )
            SELECT 
                ?,
                pf.fee_type_id,
                pf.academic_year_id,
                ft.amount,
                DATE_ADD(CURRENT_DATE, INTERVAL 30 DAY),
                'pending'
            FROM program_fees pf
            JOIN fee_types ft ON pf.fee_type_id = ft.id
            WHERE pf.program_id = ?
            AND pf.academic_year_id = ?
            AND NOT EXISTS (
                SELECT 1 
                FROM student_fees sf 
                WHERE sf.student_id = ?
                AND sf.fee_type_id = pf.fee_type_id
                AND sf.academic_year_id = pf.academic_year_id
            )
        ");
        $stmt->bind_param("iiii", $student_id, $program_id, $academic_year['id'], $student_id);
        $stmt->execute();
        
        $conn->commit();
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
}
?>