<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Content-Type: application/json');
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

try {
    // Get the payment details
    $fee_id = intval($_POST['fee_id']);
    $amount = floatval($_POST['amount']);
    $student_id = intval($_POST['student_id']);
    
    // Verify the student and fee exist
    $stmt = $conn->prepare("
        SELECT s.*, sf.amount as fee_amount, sf.id as fee_id, u.email
        FROM students s
        JOIN student_fees sf ON sf.id = ?
        JOIN users u ON s.user_id = u.id
        WHERE s.id = ?
    ");
    $stmt->bind_param("ii", $fee_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception('Invalid student or fee details');
    }
    
    $student = $result->fetch_assoc();
    
    // Generate a unique reference
    $reference = 'KCOE_' . time() . '_' . $student_id . '_' . $fee_id;
    
    // Initialize Paystack transaction
    $url = "https://api.paystack.co/transaction/initialize";
    $fields = [
        'email' => $student['email'],
        'amount' => $amount * 100, // Convert to kobo (smallest currency unit)
        'reference' => $reference,
        'callback_url' => PAYSTACK_CALLBACK_URL,
        'channels' => ['card', 'bank', 'ussd', 'qr', 'mobile_money'],
        'metadata' => [
            'student_id' => $student_id,
            'fee_id' => $fee_id,
            'is_test' => true, // For monitoring test transactions
            'custom_fields' => [
                [
                    'display_name' => "Student Name",
                    'variable_name' => "student_name",
                    'value' => $student['first_name'] . ' ' . $student['last_name']
                ],
                [
                    'display_name' => "Student Number",
                    'variable_name' => "student_number",
                    'value' => $student['student_number']
                ]
            ]
        ]
    ];
    
    $headers = [
        'Authorization: Bearer ' . PAYSTACK_SECRET_KEY,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        throw new Exception(curl_error($ch));
    }
    
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (!$result['status']) {
        throw new Exception($result['message']);
    }
    
    // Store the payment attempt in the database
    $stmt = $conn->prepare("
        INSERT INTO payments (
            student_fee_id, 
            amount, 
            payment_method, 
            reference_number, 
            status,
            payment_data
        ) VALUES (?, ?, 'paystack', ?, 'pending', ?)
    ");
    
    $payment_data = json_encode([
        'paystack_reference' => $reference,
        'authorization_url' => $result['data']['authorization_url']
    ]);
    
    $stmt->bind_param("idss", $fee_id, $amount, $reference, $payment_data);
    $stmt->execute();
    
    // Redirect to Paystack checkout page
    header('Location: ' . $result['data']['authorization_url']);
    exit;
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
?>