<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

try {
    // Get the payment details
    $fee_id = intval($_POST['fee_id']);
    $amount = floatval($_POST['amount']);
    $student_id = intval($_POST['student_id']);
    
    // Verify the student and fee exist
    $stmt = $conn->prepare("
        SELECT s.*, sf.amount as fee_amount, sf.id as fee_id, u.email
        FROM students s
        JOIN student_fees sf ON sf.id = ?
        JOIN users u ON s.user_id = u.id
        WHERE s.id = ?
    ");
    $stmt->bind_param("ii", $fee_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception('Invalid student or fee details');
    }
    
    $student = $result->fetch_assoc();
    
    // Generate a unique reference
    $reference = 'KCOE_BTF_' . time() . '_' . $student_id . '_' . $fee_id;
    
    // Initialize Bank Tranflow transaction
    $url = BANK_TRANFLOW_API_URL . '/transactions/initialize';
    $fields = [
        'merchantId' => BANK_TRANFLOW_MERCHANT_ID,
        'amount' => $amount,
        'currency' => 'GHS',
        'reference' => $reference,
        'callbackUrl' => BANK_TRANFLOW_CALLBACK_URL,
        'customerEmail' => $student['email'],
        'customerName' => $student['first_name'] . ' ' . $student['last_name'],
        'customerPhone' => $student['contact_number'] ?? '',
        'metadata' => [
            'student_id' => $student_id,
            'fee_id' => $fee_id,
            'student_number' => $student['student_number']
        ]
    ];
    
    $headers = [
        'Authorization: Bearer ' . BANK_TRANFLOW_API_KEY,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        throw new Exception(curl_error($ch));
    }
    
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (!isset($result['status']) || $result['status'] !== 'success') {
        throw new Exception($result['message'] ?? 'Failed to initialize payment');
    }
    
    // Store the payment attempt in the database
    $stmt = $conn->prepare("
        INSERT INTO payments (
            student_fee_id, 
            amount, 
            payment_method, 
            reference_number, 
            status,
            payment_data
        ) VALUES (?, ?, 'bank_tranflow', ?, 'pending', ?)
    ");
    
    $payment_data = json_encode([
        'tranflow_reference' => $reference,
        'payment_url' => $result['data']['paymentUrl']
    ]);
    
    $stmt->bind_param("idss", $fee_id, $amount, $reference, $payment_data);
    $stmt->execute();
    
    echo json_encode([
        'status' => 'success',
        'message' => 'Payment initialized successfully',
        'data' => [
            'paymentUrl' => $result['data']['paymentUrl'],
            'reference' => $reference,
            'bankDetails' => $result['data']['bankDetails'] ?? null
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
?>