<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Ensure user is logged in
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reference'])) {
    $reference = trim($_POST['reference']);
    
    // Verify with Paystack
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.paystack.co/transaction/verify/" . rawurlencode($reference),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer " . PAYSTACK_SECRET_KEY,
            "Cache-Control: no-cache",
        ],
    ]);
    
    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if ($err) {
        $error = "Failed to verify payment: " . $err;
    } else {
        $result = json_decode($response, true);
        if ($result['status'] && $result['data']['status'] === 'success') {
            // Update payment status
            $stmt = $conn->prepare("
                UPDATE payments 
                SET status = 'confirmed' 
                WHERE reference_number = ? AND status = 'pending'
            ");
            $stmt->bind_param("s", $reference);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                // Update student fee status
                $stmt = $conn->prepare("
                    UPDATE student_fees sf
                    JOIN payments p ON p.student_fee_id = sf.id
                    SET sf.status = CASE 
                        WHEN (SELECT SUM(amount) FROM payments 
                              WHERE student_fee_id = sf.id AND status = 'confirmed') >= sf.amount 
                        THEN 'paid'
                        ELSE 'partial'
                    END
                    WHERE p.reference_number = ?
                ");
                $stmt->bind_param("s", $reference);
                $stmt->execute();
                
                $message = "Payment verified and updated successfully!";
            } else {
                $error = "Payment already verified or reference number not found.";
            }
        } else {
            $error = "Payment verification failed: " . ($result['message'] ?? 'Unknown error');
        }
    }
}

$page_title = "Manual Payment Verification";
require_once '../includes/header.php';
?>

<div class="container py-4">
    <div class="row justify-content-center">
        <div class="col-md-8">
            <div class="card shadow">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Manual Payment Verification</h5>
                </div>
                <div class="card-body">
                    <?php if ($message): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?php echo $message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?php echo $error; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <form method="POST" action="" class="needs-validation" novalidate>
                        <div class="mb-3">
                            <label for="reference" class="form-label">Payment Reference Number</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="reference" 
                                   name="reference" 
                                   required 
                                   placeholder="Enter your Paystack reference number">
                            <div class="form-text">
                                Enter the reference number you received from Paystack after making the payment.
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary">Verify Payment</button>
                        <a href="../student/dashboard.php" class="btn btn-secondary">Back to Dashboard</a>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>