<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/payment_logger.php';

// Enable error reporting for debugging
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Create a log file specifically for payment verification
$log_file = __DIR__ . '/payment_verification.log';
function log_payment($message) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $message\n", FILE_APPEND);
}

// Handle Paystack verification
function verifyPaystackPayment($reference) {
    global $conn;
    log_payment("Verifying payment for reference: " . $reference);
    
    $url = "https://api.paystack.co/transaction/verify/" . urlencode($reference);
    $headers = [
        'Authorization: Bearer ' . PAYSTACK_SECRET_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    
    if ($err) {
        log_payment("Paystack API Error: " . $err);
        return false;
    }
    
    $result = json_decode($response, true);
    if (!$result['status']) {
        log_payment("Paystack API Response Error: " . ($result['message'] ?? 'Unknown error'));
        return false;
    }
    
    // Check if this transaction has been refunded
    if (isset($result['data']['status']) && $result['data']['status'] === 'reversed') {
        log_payment("Payment has been refunded");
        // Update payment status in our database
        $stmt = $conn->prepare("
            UPDATE payments 
            SET 
                status = 'refunded',
                refunded_amount = amount,
                updated_at = CURRENT_TIMESTAMP,
                payment_data = JSON_SET(
                    COALESCE(payment_data, '{}'),
                    '$.refund_status', 'completed',
                    '$.refund_date', CURRENT_TIMESTAMP
                )
            WHERE reference_number = ?
        ");
        $stmt->bind_param("s", $reference);
        $stmt->execute();
    }
    
    log_payment("Payment verification successful");
    return $result;
}

// Handle POST requests (webhooks)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    log_payment("POST request received from: " . $_SERVER['REMOTE_ADDR']);
    
    // Get the raw POST data
    $input = file_get_contents("php://input");
    log_payment("Received input: " . $input);
    
    // Parse the webhook payload
    $event = json_decode($input, true);
    if (!$event) {
        log_payment("Failed to parse webhook JSON");
        http_response_code(400);
        exit();
    }
    
    // Process the payment if we have a reference
    if (!empty($event['data']['reference'])) {
        $reference = $event['data']['reference'];
        log_payment("Processing webhook for reference: " . $reference);
        
        // Verify with Paystack API
        $verification = verifyPaystackPayment($reference);
        if ($verification && $verification['data']['status'] === 'success') {
            // Update payment status
            $stmt = $conn->prepare("
                UPDATE payments 
                SET status = 'confirmed', 
                    payment_data = ? 
                WHERE reference_number = ? 
                AND status = 'pending'
            ");
            $stmt->bind_param("ss", json_encode($verification['data']), $reference);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                // Update student fee status
                $stmt = $conn->prepare("
                    UPDATE student_fees sf
                    SET status = CASE 
                        WHEN (
                            SELECT SUM(amount) 
                            FROM payments 
                            WHERE student_fee_id = sf.id 
                            AND status = 'confirmed'
                        ) >= sf.amount THEN 'paid'
                        ELSE 'partial'
                    END
                    WHERE id = (
                        SELECT student_fee_id 
                        FROM payments 
                        WHERE reference_number = ?
                    )
                ");
                $stmt->bind_param("s", $reference);
                $stmt->execute();
                
                log_payment("Payment status updated successfully");
            } else {
                log_payment("Payment already processed or not found");
            }
        } else {
            log_payment("Payment verification failed");
        }
    }
    
    http_response_code(200);
    exit();
}

// Handle GET requests (callback URL)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $reference = $_GET['reference'] ?? '';
    
    if (empty($reference)) {
        header('Location: ../student/dashboard.php?error=invalid_reference');
        exit();
    }
    
    log_payment("Processing callback for reference: " . $reference);
    
    // Verify the payment
    $verification = verifyPaystackPayment($reference);
    if ($verification && $verification['data']['status'] === 'success') {
        // Update payment status
        $stmt = $conn->prepare("
            UPDATE payments 
            SET status = 'confirmed', 
                payment_data = ? 
            WHERE reference_number = ? 
            AND status = 'pending'
        ");
        $stmt->bind_param("ss", json_encode($verification['data']), $reference);
        
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            // Update student fee status
            $stmt = $conn->prepare("
                UPDATE student_fees sf
                SET status = CASE 
                    WHEN (
                        SELECT SUM(amount) 
                        FROM payments 
                        WHERE student_fee_id = sf.id 
                        AND status = 'confirmed'
                    ) >= sf.amount THEN 'paid'
                    ELSE 'partial'
                END
                WHERE id = (
                    SELECT student_fee_id 
                    FROM payments 
                    WHERE reference_number = ?
                )
            ");
            $stmt->bind_param("s", $reference);
            $stmt->execute();
            
            log_payment("Payment status updated successfully");
            header('Location: ../student/dashboard.php?success=payment_confirmed');
        } else {
            log_payment("Payment already processed");
            header('Location: ../student/dashboard.php?info=payment_already_confirmed');
        }
    } else {
        log_payment("Payment verification failed");
        header('Location: ../student/dashboard.php?error=payment_failed');
    }
    exit();
}

// Invalid request method
http_response_code(405);
exit();
?>