<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';

// Handle Bank Tranflow webhook verification
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Retrieve the request's body
    $input = file_get_contents("php://input");
    $event = json_decode($input, true);
    
    // Verify webhook signature
    $signature = isset($_SERVER['HTTP_X_TRANFLOW_SIGNATURE']) ? $_SERVER['HTTP_X_TRANFLOW_SIGNATURE'] : '';
    $expected_signature = hash_hmac('sha256', $input, BANK_TRANFLOW_API_KEY);
    
    if (!$signature || $signature !== $expected_signature) {
        http_response_code(400);
        exit();
    }
    
    // Handle the event
    if ($event['event'] === 'payment.success') {
        $reference = $event['data']['reference'];
        $status = strtolower($event['data']['status']);
        
        // Verify the transaction status
        $stmt = $conn->prepare("
            SELECT p.*, sf.student_id, sf.amount as fee_amount, s.user_id
            FROM payments p
            JOIN student_fees sf ON p.student_fee_id = sf.id
            JOIN students s ON sf.student_id = s.id
            WHERE p.reference_number = ? AND p.status = 'pending'
        ");
        
        $stmt->bind_param("s", $reference);
        $stmt->execute();
        $payment = $stmt->get_result()->fetch_assoc();
        
        if ($payment && $status === 'success') {
            // Start transaction
            $conn->begin_transaction();
            
            try {
                // Update payment status
                $stmt = $conn->prepare("UPDATE payments SET status = 'confirmed' WHERE reference_number = ?");
                $stmt->bind_param("s", $reference);
                $stmt->execute();
                
                // Update student fee status
                $stmt = $conn->prepare("
                    UPDATE student_fees sf
                    SET status = 
                        CASE 
                            WHEN (SELECT SUM(amount) FROM payments WHERE student_fee_id = sf.id AND status = 'confirmed') >= sf.amount 
                            THEN 'paid'
                            ELSE 'partial'
                        END
                    WHERE id = ?
                ");
                $stmt->bind_param("i", $payment['student_fee_id']);
                $stmt->execute();
                
                // Send email notification
                $stmt = $conn->prepare("
                    SELECT u.email, s.first_name, ft.name as fee_type
                    FROM users u
                    JOIN students s ON u.id = s.user_id
                    JOIN student_fees sf ON s.id = sf.student_id
                    JOIN fee_types ft ON sf.fee_type_id = ft.id
                    WHERE sf.id = ?
                ");
                $stmt->bind_param("i", $payment['student_fee_id']);
                $stmt->execute();
                $email_data = $stmt->get_result()->fetch_assoc();
                
                $email_subject = "Bank Tranflow Payment Confirmation";
                $email_message = "
                    <h2>Payment Confirmation</h2>
                    <p>Dear " . htmlspecialchars($email_data['first_name']) . ",</p>
                    <p>Your bank payment has been confirmed:</p>
                    <ul>
                        <li>Fee Type: " . htmlspecialchars($email_data['fee_type']) . "</li>
                        <li>Amount: GHs " . number_format($payment['amount'], 2) . "</li>
                        <li>Reference: " . htmlspecialchars($reference) . "</li>
                    </ul>
                    <p>Thank you for your payment.</p>
                ";
                
                sendEmailNotification($email_data['email'], $email_subject, $email_message);
                
                $conn->commit();
            } catch (Exception $e) {
                $conn->rollback();
                error_log("Error processing Bank Tranflow payment: " . $e->getMessage());
            }
        }
    }
    
    http_response_code(200);
    exit();
}

// Handle callback URL verification
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $reference = $_GET['reference'] ?? '';
    
    if (empty($reference)) {
        header('Location: ../student/dashboard.php?error=invalid_reference');
        exit();
    }
    
    // Verify the transaction status
    $url = BANK_TRANFLOW_API_URL . '/transactions/verify/' . urlencode($reference);
    $headers = [
        'Authorization: Bearer ' . BANK_TRANFLOW_API_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    
    if ($err) {
        header('Location: ../student/dashboard.php?error=verification_failed');
        exit();
    }
    
    $result = json_decode($response, true);
    
    if ($result['status'] === 'success' && $result['data']['status'] === 'success') {
        header('Location: ../student/dashboard.php?success=payment_confirmed');
    } else {
        header('Location: ../student/dashboard.php?error=payment_failed');
    }
    exit();
}

http_response_code(405);
exit();
?>