<?php
require_once 'config/config.php';
require_once 'controllers/AuthController.php';
require_once 'includes/functions.php';

session_start();

// Verify student/admin/accountant access
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get student ID from URL for admin/accountant, or from session for students
$student_id = null;
if ($_SESSION['role'] === 'student') {
    // For students, get their own ID
    $stmt = $conn->prepare("SELECT id FROM students WHERE user_id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $student_id = $result->fetch_assoc()['id'];
} else if (isset($_GET['id']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'accountant')) {
    // For admin/accountant, get ID from URL
    $student_id = intval($_GET['id']);
} else {
    header('Location: ../dashboard.php');
    exit();
}

// Get student details
$stmt = $conn->prepare("
    SELECT s.*, u.username, u.email 
    FROM students s 
    JOIN users u ON s.user_id = u.id 
    WHERE s.id = ?
");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$student = $stmt->get_result()->fetch_assoc();

// Get payment history grouped by year level and academic year
$stmt = $conn->prepare("
    SELECT 
        sf.student_id,
        s.year_level as payment_year_level,
        ay.year_name as academic_year,
        ft.name as fee_type,
        sf.amount as fee_amount,
        sf.due_date,
        sf.status as fee_status,
        p.id as payment_id,
        p.amount as paid_amount,
        p.payment_date,
        p.payment_method,
        p.reference_number,
        p.status as payment_status,
        u.username as processed_by
    FROM student_fees sf
    JOIN students s ON sf.student_id = s.id
    JOIN academic_years ay ON sf.academic_year_id = ay.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    LEFT JOIN payments p ON sf.id = p.student_fee_id
    LEFT JOIN users u ON p.processed_by = u.id
    WHERE sf.student_id = ?
    ORDER BY s.year_level ASC, ay.start_date ASC, ft.name ASC
");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();

// Group payments by year level
$payments_by_year = [];
while ($row = $result->fetch_assoc()) {
    $year_level = $row['payment_year_level'];
    if (!isset($payments_by_year[$year_level])) {
        $payments_by_year[$year_level] = [
            'academic_year' => $row['academic_year'],
            'fees' => [],
            'total_fees' => 0,
            'total_paid' => 0
        ];
    }
    
    $fee_key = $row['fee_type'];
    if (!isset($payments_by_year[$year_level]['fees'][$fee_key])) {
        $payments_by_year[$year_level]['fees'][$fee_key] = [
            'amount' => $row['fee_amount'],
            'status' => $row['fee_status'],
            'payments' => []
        ];
        $payments_by_year[$year_level]['total_fees'] += $row['fee_amount'];
    }
    
    if ($row['payment_id']) {
        $payments_by_year[$year_level]['fees'][$fee_key]['payments'][] = [
            'amount' => $row['paid_amount'],
            'date' => $row['payment_date'],
            'method' => $row['payment_method'],
            'reference' => $row['reference_number'],
            'status' => $row['payment_status'],
            'processed_by' => $row['processed_by']
        ];
        if ($row['payment_status'] === 'confirmed') {
            $payments_by_year[$year_level]['total_paid'] += $row['paid_amount'];
        }
    }
}

$page_title = "Payment History";
require_once 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4">
                <div class="card-header pb-0">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6>Student Payment History</h6>
                        <button onclick="window.print()" class="btn btn-primary btn-sm">
                            <i class="fas fa-print"></i> Print History
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Student Information -->
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h5>Student Details</h5>
                            <p class="mb-1"><strong>Name:</strong> <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></p>
                            <p class="mb-1"><strong>Student Number:</strong> <?php echo htmlspecialchars($student['student_number']); ?></p>
                            <p class="mb-1"><strong>Course:</strong> <?php echo htmlspecialchars($student['course']); ?></p>
                            <p class="mb-1"><strong>Current Year Level:</strong> <?php echo htmlspecialchars($student['year_level']); ?></p>
                        </div>
                    </div>

                    <!-- Payment History by Year Level -->
                    <?php foreach ($payments_by_year as $year_level => $year_data): ?>
                    <div class="year-level-section mb-4">
                        <h5 class="bg-light p-3 rounded">
                            <?php echo getYearLevelText($year_level); ?> 
                            (<?php echo htmlspecialchars($year_data['academic_year']); ?>)
                        </h5>
                        
                        <!-- Summary for this year level -->
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <div class="card bg-primary text-white">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Fees</h6>
                                        <h4 class="mb-0"><?php echo formatPeso($year_data['total_fees']); ?></h4>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-success text-white">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Paid</h6>
                                        <h4 class="mb-0"><?php echo formatPeso($year_data['total_paid']); ?></h4>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-info text-white">
                                    <div class="card-body">
                                        <h6 class="card-title">Balance</h6>
                                        <h4 class="mb-0"><?php echo formatPeso($year_data['total_fees'] - $year_data['total_paid']); ?></h4>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Detailed fees and payments -->
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Fee Type</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Payment Details</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($year_data['fees'] as $fee_type => $fee_data): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($fee_type); ?></td>
                                        <td><?php echo formatMoney($fee_data['amount']); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo getStatusColor($fee_data['status']); ?>">
                                                <?php echo ucfirst($fee_data['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (empty($fee_data['payments'])): ?>
                                                <em>No payments recorded</em>
                                            <?php else: ?>
                                                <?php foreach ($fee_data['payments'] as $payment): ?>
                                                    <div class="payment-record mb-2">
                                                        <strong><?php echo formatMoney($payment['amount']); ?></strong>
                                                        <br>
                                                        <small>
                                                            <?php echo date('M j, Y', strtotime($payment['date'])); ?> 
                                                            via <?php echo ucfirst($payment['method']); ?>
                                                            <?php if ($payment['reference']): ?>
                                                                (Ref: <?php echo htmlspecialchars($payment['reference']); ?>)
                                                            <?php endif; ?>
                                                            <br>
                                                            Status: 
                                                            <span class="badge bg-<?php echo getStatusColor($payment['status']); ?>">
                                                                <?php echo ucfirst($payment['status']); ?>
                                                            </span>
                                                            <?php if ($payment['processed_by']): ?>
                                                                | Processed by: <?php echo htmlspecialchars($payment['processed_by']); ?>
                                                            <?php endif; ?>
                                                        </small>
                                                    </div>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php endforeach; ?>

                    <?php if (empty($payments_by_year)): ?>
                    <div class="alert alert-info">
                        No payment records found.
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style media="print">
    .navbar, .btn, .no-print {
        display: none !important;
    }
    .card {
        border: none !important;
        box-shadow: none !important;
    }
    .year-level-section {
        page-break-inside: avoid;
    }
</style>

<?php 
function getYearLevelText($year) {
    $suffixes = ['st', 'nd', 'rd', 'th', 'th'];
    return $year . $suffixes[$year - 1] . ' Year';
}

function getStatusColor($status) {
    switch ($status) {
        case 'paid':
        case 'confirmed':
            return 'success';
        case 'pending':
            return 'warning';
        case 'overdue':
            return 'danger';
        case 'rejected':
            return 'secondary';
        default:
            return 'info';
    }
}

require_once 'includes/footer.php'; 
?>