<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/payment_logger.php';

// Set headers for JSON response
header('Content-Type: application/json');

try {
    // Get the reference from the callback
    $reference = $_GET['reference'] ?? null;
    if (!$reference) {
        throw new Exception('No reference provided');
    }
    
    // Verify the transaction with Paystack
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.paystack.co/transaction/verify/" . rawurlencode($reference),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer " . PAYSTACK_SECRET_KEY,
            "Cache-Control: no-cache",
        ],
    ]);
    
    $response = curl_exec($curl);
    if (curl_errno($curl)) {
        throw new Exception(curl_error($curl));
    }
    curl_close($curl);
    
    $result = json_decode($response, true);
    if (!$result['status']) {
        throw new Exception($result['message']);
    }
    
    // Log the callback verification
    logPaymentTransaction([
        'reference' => $reference,
        'type' => 'callback_verification',
        'method' => 'paystack',
        'amount' => $result['data']['amount'] / 100, // Convert from kobo to naira
        'status' => $result['data']['status'],
        'response' => $result
    ]);
    
    // If payment is successful, redirect to success page
    if ($result['data']['status'] === 'success') {
        // Get the student's payment details
        $stmt = $conn->prepare("
            SELECT 
                p.*,
                s.user_id,
                sf.student_id
            FROM payments p
            JOIN student_fees sf ON p.student_fee_id = sf.id
            JOIN students s ON sf.student_id = s.id
            WHERE p.reference_number = ?
        ");
        
        $stmt->bind_param("s", $reference);
        $stmt->execute();
        $payment = $stmt->get_result()->fetch_assoc();
        
        if ($payment) {
            // Redirect to payment success page
            $success_url = BASE_URL . "/student/payment_success.php?reference=" . urlencode($reference);
            echo json_encode([
                'status' => 'success',
                'message' => 'Payment verified successfully',
                'redirect_url' => $success_url
            ]);
        } else {
            throw new Exception('Payment record not found');
        }
    } else {
        // Redirect to payment failed page
        $error_url = BASE_URL . "/student/payment_failed.php?reference=" . urlencode($reference);
        echo json_encode([
            'status' => 'error',
            'message' => 'Payment verification failed',
            'redirect_url' => $error_url
        ]);
    }
} catch (Exception $e) {
    // Log the error
    logPaymentTransaction([
        'reference' => $reference ?? 'unknown',
        'type' => 'callback_error',
        'method' => 'paystack',
        'status' => 'error',
        'response' => ['error' => $e->getMessage()]
    ]);
    
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage(),
        'redirect_url' => BASE_URL . "/student/payment_failed.php"
    ]);
}