<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/payment_logger.php';
require_once '../includes/debug_logger.php';

// Set headers for webhook response
header('HTTP/1.1 200 OK');
header('Content-Type: application/json');

// Log all request headers for debugging
debugLog('Webhook Request Headers:', getallheaders());

try {
    // Get the payload from Paystack
    $input = file_get_contents('php://input');
    debugLog('Webhook Raw Payload:', $input);
    
    // Verify Paystack webhook signature
    $signature = isset($_SERVER['HTTP_X_PAYSTACK_SIGNATURE']) ? $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] : '';
    debugLog('Paystack Signature:', $signature);
    
    $calculated_signature = hash_hmac('sha512', $input, PAYSTACK_SECRET_KEY);
    debugLog('Calculated Signature:', $calculated_signature);
    
    if (!$signature || $signature !== $calculated_signature) {
        debugLog('Signature Verification Failed');
        throw new Exception('Invalid signature');
    }
    
    debugLog('Signature Verification Successful');
    
    // Parse the webhook payload
    $event = json_decode($input, true);
    if (!$event) {
        throw new Exception('Invalid JSON payload');
    }

    // Log the webhook data
    logPaymentTransaction([
        'type' => 'webhook_received',
        'method' => 'paystack',
        'reference' => $event['data']['reference'] ?? 'no_ref',
        'status' => 'received',
        'response' => $event
    ]);
    
    // Process only successful charges
    if ($event['event'] === 'charge.success') {
        $reference = $event['data']['reference'];
        
        // Get payment details from database
        $stmt = $conn->prepare("
            SELECT 
                p.*, 
                sf.student_id,
                s.first_name,
                s.last_name,
                s.user_id,
                u.email,
                ft.name as fee_type
            FROM payments p
            JOIN student_fees sf ON p.student_fee_id = sf.id
            JOIN students s ON sf.student_id = s.id
            JOIN users u ON s.user_id = u.id
            JOIN fee_types ft ON sf.fee_type_id = ft.id
            WHERE p.reference_number = ? AND p.status = 'pending'
        ");
        
        $stmt->bind_param("s", $reference);
        $stmt->execute();
        $payment = $stmt->get_result()->fetch_assoc();
        
        if ($payment) {
            // Start transaction
            $conn->begin_transaction();
            
            try {
                // Update payment status
                $stmt = $conn->prepare("
                    UPDATE payments 
                    SET 
                        status = 'confirmed',
                        updated_at = NOW(),
                        payment_date = COALESCE(payment_date, NOW())
                    WHERE id = ?
                ");
                $stmt->bind_param("i", $payment['id']);
                $stmt->execute();
                
                // Update student fee status
                $stmt = $conn->prepare("
                    UPDATE student_fees sf
                    SET status = CASE 
                        WHEN (
                            SELECT COALESCE(SUM(amount), 0) 
                            FROM payments 
                            WHERE student_fee_id = sf.id AND status = 'confirmed'
                        ) >= (
                            SELECT amount 
                            FROM fee_types 
                            WHERE id = sf.fee_type_id
                        )
                        THEN 'paid'
                        ELSE 'partial'
                    END
                    WHERE id = ?
                ");
                $stmt->bind_param("i", $payment['student_fee_id']);
                $stmt->execute();
                
                // Log the successful payment
                logPaymentTransaction([
                    'payment_id' => $payment['id'],
                    'reference' => $reference,
                    'type' => 'payment_confirmed',
                    'method' => 'paystack',
                    'amount' => $payment['amount'],
                    'status' => 'confirmed',
                    'response' => $event['data']
                ]);
                
                // Send email notification
                $email_subject = "Payment Confirmation - " . APP_NAME;
                $email_message = "
                    <h2>Payment Confirmation</h2>
                    <p>Dear " . htmlspecialchars($payment['first_name']) . ",</p>
                    <p>Your payment has been confirmed:</p>
                    <ul>
                        <li>Fee Type: " . htmlspecialchars($payment['fee_type']) . "</li>
                        <li>Amount: " . formatMoney($payment['amount']) . "</li>
                        <li>Reference: " . htmlspecialchars($reference) . "</li>
                        <li>Date: " . date('Y-m-d H:i:s') . "</li>
                    </ul>
                    <p>Thank you for your payment.</p>
                ";
                
                sendEmailNotification($payment['email'], $email_subject, $email_message);
                
                // Commit transaction
                $conn->commit();
                
                echo json_encode(['status' => 'success', 'message' => 'Payment processed successfully']);
            } catch (Exception $e) {
                $conn->rollback();
                throw $e;
            }
        } else {
            // Payment not found or already processed
            logPaymentTransaction([
                'reference' => $reference,
                'type' => 'payment_not_found',
                'method' => 'paystack',
                'status' => 'error',
                'response' => ['error' => 'Payment not found or already processed']
            ]);
            
            echo json_encode(['status' => 'error', 'message' => 'Payment not found or already processed']);
        }
    } else {
        // Log non-success events
        logPaymentTransaction([
            'reference' => $event['data']['reference'] ?? 'no_ref',
            'type' => 'other_webhook_event',
            'method' => 'paystack',
            'status' => 'received',
            'response' => $event
        ]);
        
        echo json_encode(['status' => 'success', 'message' => 'Event received']);
    }
} catch (Exception $e) {
    // Log the error
    logPaymentTransaction([
        'type' => 'webhook_error',
        'method' => 'paystack',
        'status' => 'error',
        'response' => ['error' => $e->getMessage()]
    ]);
    
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}