<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/debug_logger.php';

header('Content-Type: application/json');

try {
    // Get all Paystack payments to display
    $stmt = $conn->prepare("
        SELECT 
            p.id,
            p.reference_number,
            p.amount,
            p.status,
            p.created_at,
            s.first_name,
            s.last_name,
            ft.name as fee_type
        FROM payments p
        JOIN student_fees sf ON p.student_fee_id = sf.id
        JOIN students s ON sf.student_id = s.id
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        WHERE p.payment_method = 'paystack'
        ORDER BY p.created_at DESC
    ");
    
    $stmt->execute();
    $all_payments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
    // If no payment ID is provided, show the selection form
    if (!isset($_POST['payment_id'])) {
        echo '<html>
        <head>
            <title>Simulate Payment Webhook</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
        </head>
        <body>
            <div class="container mt-5">
                <h2>Select Payment to Simulate</h2>
                <form method="POST" class="mb-4">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Select</th>
                                    <th>Reference</th>
                                    <th>Student</th>
                                    <th>Fee Type</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>';
                            foreach ($all_payments as $payment) {
                                echo "<tr>
                                    <td><input type='radio' name='payment_id' value='{$payment['id']}' required></td>
                                    <td>{$payment['reference_number']}</td>
                                    <td>{$payment['first_name']} {$payment['last_name']}</td>
                                    <td>{$payment['fee_type']}</td>
                                    <td>" . number_format($payment['amount'], 2) . "</td>
                                    <td>{$payment['status']}</td>
                                    <td>{$payment['created_at']}</td>
                                </tr>";
                            }
                            echo '</tbody>
                        </table>
                    </div>
                    <button type="submit" class="btn btn-primary">Simulate Payment Webhook</button>
                </form>
            </div>
        </body>
        </html>';
        exit;
    }
    
    // Get the selected payment details
    $stmt = $conn->prepare("
        SELECT 
            p.*, 
            sf.student_id,
            s.first_name,
            s.last_name,
            u.email,
            ft.name as fee_type
        FROM payments p
        JOIN student_fees sf ON p.student_fee_id = sf.id
        JOIN students s ON sf.student_id = s.id
        JOIN users u ON s.user_id = u.id
        JOIN fee_types ft ON sf.fee_type_id = ft.id
        WHERE p.id = ?
    ");
    
    $stmt->bind_param("i", $_POST['payment_id']);
    $stmt->execute();
    $payment = $stmt->get_result()->fetch_assoc();
    
    $stmt->execute();
    $payment = $stmt->get_result()->fetch_assoc();
    
    if (!$payment) {
        throw new Exception('Selected payment not found.');
    }
    
    // Set header to JSON only if it's not the form display
    header('Content-Type: application/json');
    
    // Create a simulated Paystack webhook payload
    $test_payload = [
        "event" => "charge.success",
        "data" => [
            "id" => rand(1000000, 9999999),
            "reference" => $payment['reference_number'],
            "amount" => $payment['amount'] * 100, // Convert to kobo
            "gateway_response" => "Successful",
            "paid_at" => date("Y-m-d\TH:i:s.000\Z"),
            "created_at" => date("Y-m-d\TH:i:s.000\Z"),
            "channel" => "card",
            "currency" => "NGN",
            "ip_address" => "127.0.0.1",
            "metadata" => [
                "custom_fields" => [
                    [
                        "display_name" => "Student Name",
                        "variable_name" => "student_name",
                        "value" => $payment['first_name'] . ' ' . $payment['last_name']
                    ]
                ]
            ],
            "log" => null,
            "fees" => null,
            "customer" => [
                "id" => rand(1000, 9999),
                "first_name" => $payment['first_name'],
                "last_name" => $payment['last_name'],
                "email" => $payment['email'],
                "phone" => "",
                "metadata" => null,
                "risk_action" => "default"
            ],
            "authorization" => [
                "authorization_code" => "AUTH_" . strtoupper(bin2hex(random_bytes(8))),
                "bin" => "408408",
                "last4" => "4081",
                "exp_month" => "12",
                "exp_year" => "2025",
                "card_type" => "visa",
                "bank" => "TEST BANK",
                "country_code" => "NG",
                "brand" => "visa",
                "account_name" => $payment['first_name'] . ' ' . $payment['last_name']
            ],
            "plan" => null,
            "requested_amount" => $payment['amount'] * 100,
            "status" => "success",
        ]
    ];

    $json_payload = json_encode($test_payload);
    
    // Calculate signature as Paystack would
    $signature = hash_hmac('sha512', $json_payload, PAYSTACK_SECRET_KEY);
    
    echo json_encode([
        'status' => 'info',
        'message' => 'Preparing to send test webhook...',
        'payment_details' => [
            'reference' => $payment['reference_number'],
            'amount' => $payment['amount'],
            'student' => $payment['first_name'] . ' ' . $payment['last_name']
        ]
    ]);
    
    // Send test webhook to your endpoint
    $ch = curl_init(PAYSTACK_WEBHOOK_URL);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json_payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'X-Paystack-Signature: ' . $signature,
        'User-Agent: Paystack-Webhook-Test'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        throw new Exception('Curl error: ' . curl_error($ch));
    }
    
    curl_close($ch);
    
    // Check webhook response
    if ($http_code === 200) {
        // Verify the payment was processed
        $stmt = $conn->prepare("SELECT status FROM payments WHERE reference_number = ?");
        $stmt->bind_param("s", $payment['reference_number']);
        $stmt->execute();
        $updated_payment = $stmt->get_result()->fetch_assoc();
        
        if ($updated_payment && $updated_payment['status'] === 'confirmed') {
            echo json_encode([
                'status' => 'success',
                'message' => 'Payment simulation completed successfully!',
                'details' => [
                    'http_code' => $http_code,
                    'webhook_response' => json_decode($response, true),
                    'payment_status' => 'confirmed',
                    'reference' => $payment['reference_number']
                ]
            ]);
        } else {
            echo json_encode([
                'status' => 'warning',
                'message' => 'Webhook received but payment not confirmed',
                'details' => [
                    'http_code' => $http_code,
                    'webhook_response' => json_decode($response, true),
                    'payment_status' => $updated_payment['status'] ?? 'unknown',
                    'reference' => $payment['reference_number']
                ]
            ]);
        }
    } else {
        throw new Exception("Webhook failed with HTTP code: $http_code");
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}