<?php
require_once '../config/config.php';
require_once '../config/payment_config.php';
require_once '../includes/functions.php';
require_once '../includes/debug_logger.php';

function updatePaymentStatus($payment_id, $status = 'confirmed') {
    global $conn;
    
    try {
        $conn->begin_transaction();
        
        // Update payment status
        $stmt = $conn->prepare("
            UPDATE payments 
            SET 
                status = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->bind_param("si", $status, $payment_id);
        $stmt->execute();
        
        // Update student fee status
        $stmt = $conn->prepare("
            UPDATE student_fees sf
            JOIN payments p ON p.student_fee_id = sf.id
            SET sf.status = CASE 
                WHEN (
                    SELECT COALESCE(SUM(amount), 0) 
                    FROM payments 
                    WHERE student_fee_id = sf.id 
                    AND status = 'confirmed'
                ) >= (
                    SELECT amount 
                    FROM fee_types ft 
                    WHERE ft.id = sf.fee_type_id
                )
                THEN 'paid'
                ELSE 'partial'
            END
            WHERE p.id = ?
        ");
        $stmt->bind_param("i", $payment_id);
        $stmt->execute();
        
        $conn->commit();
        return true;
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
}

if (isset($_GET['action']) && $_GET['action'] === 'update_status' && isset($_GET['payment_id'])) {
    header('Content-Type: application/json');
    try {
        $payment_id = intval($_GET['payment_id']);
        $status = $_GET['status'] ?? 'confirmed';
        
        if (updatePaymentStatus($payment_id, $status)) {
            echo json_encode(['status' => 'success', 'message' => 'Payment status updated successfully']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Failed to update payment status']);
        }
        exit;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        exit;
    }
}

// Display all payments
$stmt = $conn->prepare("
    SELECT 
        p.id,
        p.reference_number,
        p.amount,
        p.status,
        p.payment_method,
        p.created_at,
        p.updated_at,
        s.first_name,
        s.last_name,
        ft.name as fee_type
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    ORDER BY p.created_at DESC
");

$stmt->execute();
$payments = $stmt->get_result();
?>

<!DOCTYPE html>
<html>
<head>
    <title>Payment Status Manager</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</head>
<body>
    <div class="container mt-5">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h2 class="mb-0">Payment Status Manager</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Reference</th>
                                <th>Student</th>
                                <th>Fee Type</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Created</th>
                                <th>Last Updated</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($payment = $payments->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($payment['reference_number']); ?></td>
                                    <td><?php echo htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']); ?></td>
                                    <td><?php echo htmlspecialchars($payment['fee_type']); ?></td>
                                    <td><?php echo formatMoney($payment['amount']); ?></td>
                                    <td><?php echo ucfirst($payment['payment_method'] ?? '-'); ?></td>
                                    <td><?php echo date('Y-m-d H:i:s', strtotime($payment['created_at'])); ?></td>
                                    <td><?php echo date('Y-m-d H:i:s', strtotime($payment['updated_at'])); ?></td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            echo $payment['status'] === 'confirmed' ? 'success' : 
                                                ($payment['status'] === 'pending' ? 'warning' : 'danger'); 
                                        ?>">
                                            <?php echo ucfirst($payment['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($payment['status'] !== 'confirmed'): ?>
                                            <button 
                                                class="btn btn-sm btn-success"
                                                onclick="updateStatus(<?php echo $payment['id']; ?>, 'confirmed')"
                                            >
                                                Confirm
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script>
    function updateStatus(paymentId, status) {
        if (!confirm('Are you sure you want to update this payment status?')) {
            return;
        }
        
        $.get('update_payment_status.php', {
            action: 'update_status',
            payment_id: paymentId,
            status: status
        })
        .done(function(response) {
            if (response.status === 'success') {
                alert('Payment status updated successfully');
                location.reload();
            } else {
                alert('Error: ' + response.message);
            }
        })
        .fail(function() {
            alert('Error updating payment status');
        });
    }
    </script>
</body>
</html>