<?php
// Start session and include required files
session_start();
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Initialize database connection
try {
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        throw new Exception("Connection failed: " . $conn->connect_error);
    }
    $conn->set_charset("utf8mb4");
} catch (Exception $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Verify student access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student') {
    header('Location: ../login.php');
    exit();
}

// Initialize variables
$password_message = '';
$student = [];
$financials = [
    'total_fees' => 0,
    'total_payments' => 0,
    'provisional_fees' => 0,
    'year_payments' => 0,
    'existing_balance' => 0,
    'balance_forward' => 0
];
$current_academic_year = '2025/2026';

// Handle password reset
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'reset_password') {
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $password_message = 'All password fields are required.';
    } elseif ($new_password !== $confirm_password) {
        $password_message = 'New passwords do not match.';
    } elseif (strlen($new_password) < 8) {
        $password_message = 'Password must be at least 8 characters long.';
    } else {
        // Verify current password
        $stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $user = $stmt->get_result()->fetch_assoc();

        if (!password_verify($current_password, $user['password'])) {
            $password_message = 'Current password is incorrect.';
        } else {
            // Update password
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);
            $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->bind_param("si", $new_password_hash, $_SESSION['user_id']);
            
            $password_message = $stmt->execute() ? 'Password updated successfully!' : 'Error updating password.';
        }
    }
}

// Get student information
try {
    // Get student basic information
    $query = "
        SELECT 
            s.id,
            s.student_number,
            s.first_name,
            s.last_name,
            s.year_level,
            u.email,
            p.program_name,
            COALESCE(ay.year_name, 'Not Set') as academic_year
        FROM students s
        INNER JOIN users u ON s.user_id = u.id
        LEFT JOIN programs p ON s.program_id = p.id
        LEFT JOIN academic_years ay ON ay.is_current = 1
        WHERE s.user_id = ?";

    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();

    if (!$student) {
        throw new Exception("Student record not found");
    }

    // Get financial information
    $financial_queries = [
        'total_fees' => "SELECT COALESCE(SUM(amount), 0) as amount FROM student_fees WHERE student_id = ?",
        'total_payments' => "SELECT COALESCE(SUM(p.amount), 0) as amount FROM payments p JOIN student_fees sf ON p.student_fee_id = sf.id WHERE sf.student_id = ? AND p.status = 'confirmed'",
        'provisional_fees' => "SELECT COALESCE(SUM(sf.amount), 0) as amount FROM student_fees sf JOIN academic_years ay ON sf.academic_year_id = ay.id WHERE sf.student_id = ? AND ay.is_current = 1",
        'year_payments' => "SELECT COALESCE(SUM(p.amount), 0) as amount FROM payments p JOIN student_fees sf ON p.student_fee_id = sf.id WHERE sf.student_id = ? AND p.status = 'confirmed' AND YEAR(p.payment_date) = 2025"
    ];

    foreach ($financial_queries as $key => $query) {
        $stmt = $conn->prepare($query);
        $stmt->bind_param("i", $student['id']);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $financials[$key] = $result['amount'];
    }

    // Calculate balances
    $financials['existing_balance'] = $financials['total_fees'] - $financials['total_payments'];
    $financials['balance_forward'] = $financials['existing_balance'] + $financials['provisional_fees'];

} catch (Exception $e) {
    die("Error: " . $e->getMessage());
}

$page_title = "Student Dashboard";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <!-- Personal Details Card -->
        <div class="col-md-6 mb-4">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Personal Details</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="text-muted">Student Number</label>
                        <h6><?php echo htmlspecialchars($student['student_number'] ?? 'Not Assigned'); ?></h6>
                    </div>
                    <div class="mb-3">
                        <label class="text-muted">Name</label>
                        <h6><?php echo htmlspecialchars(($student['first_name'] ?? '') . ' ' . ($student['last_name'] ?? '')); ?></h6>
                    </div>
                    <div class="mb-3">
                        <label class="text-muted">Program</label>
                        <h6><?php echo htmlspecialchars($student['program_name'] ?? 'Not Assigned'); ?></h6>
                    </div>
                    <div class="mb-3">
                        <label class="text-muted">Year Level</label>
                        <h6><?php echo htmlspecialchars($student['year_level'] ?? 'Not Set'); ?></h6>
                    </div>
                    <div class="mb-3">
                        <label class="text-muted">Email</label>
                        <h6><?php echo htmlspecialchars($student['email'] ?? 'Not Set'); ?></h6>
                    </div>
                    <div class="mb-3">
                        <label class="text-muted">Academic Year</label>
                        <h6><?php echo htmlspecialchars($student['academic_year'] ?? 'Not Set'); ?></h6>
                    </div>
                </div>
            </div>
        </div>

        <!-- Financial Summary Card -->
        <div class="col-md-6 mb-4">
            <div class="card">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Financial Summary</h5>
                    <span class="badge bg-light text-primary">Currency: GH₵</span>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <tbody>
                                <tr>
                                    <td class="text-muted">Existing Balance</td>
                                    <td class="text-end fw-bold"><?php echo formatMoney($financials['existing_balance']); ?></td>
                                </tr>
                                <tr>
                                    <td class="text-muted">Provisional Fees (<?php echo $current_academic_year; ?>)</td>
                                    <td class="text-end fw-bold"><?php echo formatMoney($financials['provisional_fees']); ?></td>
                                </tr>
                                <tr>
                                    <td class="text-muted">Payments Made This Year</td>
                                    <td class="text-end fw-bold"><?php echo formatMoney($financials['year_payments']); ?></td>
                                </tr>
                                <tr>
                                    <td class="text-muted">Total Payments</td>
                                    <td class="text-end fw-bold"><?php echo formatMoney($financials['total_payments']); ?></td>
                                </tr>
                                <tr class="table-info">
                                    <td class="fw-bold">Balance Carry Forward</td>
                                    <td class="text-end fw-bold"><?php echo formatMoney($financials['balance_forward']); ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Reset Password Card -->
        <div class="col-md-6 mb-4">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Reset Password</h5>
                </div>
                <div class="card-body">
                    <?php if ($password_message): ?>
                        <div class="alert alert-info"><?php echo htmlspecialchars($password_message); ?></div>
                    <?php endif; ?>
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="reset_password">
                        <div class="mb-3">
                            <label for="current_password" class="form-label">Current Password</label>
                            <input type="password" class="form-control" id="current_password" name="current_password" required>
                        </div>
                        <div class="mb-3">
                            <label for="new_password" class="form-label">New Password</label>
                            <input type="password" class="form-control" id="new_password" name="new_password" required>
                        </div>
                        <div class="mb-3">
                            <label for="confirm_password" class="form-label">Confirm New Password</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Update Password</button>
                    </form>
                </div>
            </div>
        </div>

        <!-- Fee Details & Payment History -->
        <div class="col-md-6 mb-4">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Fee Details & Payment History</h5>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <a href="fee_statement.php" class="btn btn-info">View Fee Statement</a>
                        <a href="payments.php" class="btn btn-primary">Make Payment</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>