<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';
require_once '../includes/due_date_functions.php';
require_once '../includes/financial_functions.php';

session_start();

// Verify student access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student') {
    header('Location: ../login.php');
    exit();
}

// Get student details
$stmt = $conn->prepare("
    SELECT 
        s.*,
        u.email,
        p.program_name,
        (SELECT year_name FROM academic_years WHERE is_current = 1) as current_year
    FROM students s
    JOIN users u ON s.user_id = u.id
    LEFT JOIN programs p ON s.program_id = p.id
    WHERE s.user_id = ?
");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$student = $stmt->get_result()->fetch_assoc();

// Get current academic year
$stmt = $conn->prepare("SELECT id, year_name FROM academic_years WHERE is_current = 1");
$stmt->execute();
$current_academic_year = $stmt->get_result()->fetch_assoc();

// Get fee details grouped by level with academic year info and payment history
$stmt = $conn->prepare("
    SELECT 
        sf.*,
        ft.name as fee_type,
        ft.level,
        ay.year_name,
        ay.id as academic_year_id,
        (SELECT COALESCE(SUM(amount), 0) 
         FROM payments 
         WHERE student_fee_id = sf.id AND status = 'confirmed') as paid_amount,
        (SELECT GROUP_CONCAT(
            CONCAT(
                DATE_FORMAT(payment_date, '%Y-%m-%d %H:%i:%s'),
                '|',
                amount,
                '|',
                UPPER(payment_method),
                '|',
                reference_number,
                '|',
                status
            ) SEPARATOR ';;'
         ) FROM payments 
         WHERE student_fee_id = sf.id 
         ORDER BY payment_date DESC) as payment_history,
        (SELECT MAX(payment_date) 
         FROM payments 
         WHERE student_fee_id = sf.id AND status = 'confirmed') as last_payment_date
    FROM student_fees sf
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    JOIN academic_years ay ON sf.academic_year_id = ay.id
    WHERE sf.student_id = ?
    ORDER BY ft.level ASC, ay.year_name DESC
");
$stmt->bind_param("i", $student['id']);
$stmt->execute();
$fees = $stmt->get_result();



// Get financial summary using the shared function
$financial_summary = calculateStudentFinancials($conn, $student['id']);

// Calculate totals
$total_fees = 0;
$total_paid = 0;
$total_balance = 0;

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fee Statement - <?php echo htmlspecialchars($student['student_number']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print {
                display: none !important;
            }
            .page-break {
                page-break-before: always;
            }
            body {
                padding: 20px !important;
            }
        }
        .statement-header {
            text-align: center;
            margin-bottom: 40px;
            padding: 20px;
        }
        .letterhead {
            max-width: 800px;
            margin: 0 auto;
        }
        .logo-container {
            margin-bottom: 20px;
        }
        .table th {
            background-color: #f8f9fa !important;
        }
        .footer {
            margin-top: 50px;
            border-top: 1px solid #dee2e6;
            padding-top: 20px;
        }
    </style>
</head>
<body class="bg-white">
    <div class="container py-4">
        <div class="no-print mb-4">
            <button onclick="window.print()" class="btn btn-primary">
                <i class="fas fa-print"></i> Print Statement
            </button>
            <button onclick="refreshStatement()" class="btn btn-success">
                <i class="fas fa-sync"></i> Refresh Statement
            </button>
            <a href="dashboard.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to Dashboard
            </a>
            <div class="form-check form-switch d-inline-block ms-3">
                <input class="form-check-input" type="checkbox" id="autoRefresh" onchange="toggleAutoRefresh(this)">
                <label class="form-check-label" for="autoRefresh">Auto-refresh every 5 minutes</label>
            </div>
            <div id="lastUpdated" class="text-muted small d-inline-block ms-3">
                Last updated: <?php echo date('h:i A'); ?>
            </div>
        </div>

        <div class="statement-header">
            <div class="letterhead">
                <div class="logo-container">
                    <img src="../assets/images/logo.png" alt="KCOE Logo" class="header-logo" style="height: 120px; width: auto;">
                </div>
                <div class="institution-info">
                    <h3 style="color: #800000; font-size: 28px; font-weight: bold; letter-spacing: 0.5px; margin: 20px 0 10px; font-family: 'Times New Roman', Times, serif;">
                        KOMENDA COLLEGE OF EDUCATION
                    </h3>
                    <p style="color: #444; font-size: 15px; margin-bottom: 5px;">P.O. Box KM 5, Komenda</p>
                    <p style="color: #444; font-size: 15px; margin-bottom: 20px;">Cape Coast, Ghana</p>
                </div>
                <div style="border-bottom: 2px solid #800000; margin: 20px auto; width: 80%; max-width: 500px;"></div>
                <h4 style="font-size: 20px; text-transform: uppercase; letter-spacing: 1px; color: #333; margin: 20px 0;">
                    OFFICIAL FEE STATEMENT
                </h4>
                <p style="font-size: 14px; color: #666; font-style: italic;">
                    Generated on <?php echo date('F d, Y'); ?>
                </p>
            </div>
        </div>

        <style>
            @page {
                margin: 2cm;
            }
            .statement-header {
                text-align: center;
                margin-bottom: 40px;
                padding: 20px;
            }
            .letterhead {
                width: 100%;
                max-width: 800px;
                margin: 0 auto;
            }
            .logo-container {
                text-align: center;
                margin-bottom: 20px;
            }
            .header-logo {
                height: 120px;
                width: auto;
            }
            .institution-name {
                color: #800000;
                font-family: 'Times New Roman', Times, serif;
                font-size: 24px;
                font-weight: bold;
                margin: 15px 0 10px;
                text-align: center;
            }
            .institution-address {
                color: #444;
                font-size: 14px;
                line-height: 1.4;
                margin: 5px 0;
                text-align: center;
            }
            .document-title {
                margin: 30px 0 15px;
                padding-bottom: 8px;
                border-bottom: 2px solid #800000;
                text-align: center;
            }
            .document-title h4 {
                color: #333;
                font-size: 18px;
                font-weight: 600;
                letter-spacing: 1px;
                text-transform: uppercase;
                margin: 0;
            }
            .generation-date {
                color: #666;
                font-style: italic;
                text-align: center;
                font-size: 13px;
                margin-top: 10px;
            }
            @media print {
                .statement-header {
                    background: none;
                    box-shadow: none;
                }
            }
        </style>

        <div class="row mb-4">
            <div class="col-md-6">
                <h5>Student Information</h5>
                <table class="table table-bordered">
                    <tr>
                        <th width="40%">Student Number</th>
                        <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                    </tr>
                    <tr>
                        <th>Name</th>
                        <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></td>
                    </tr>
                    <tr>
                        <th>Program</th>
                        <td><?php echo htmlspecialchars($student['program_name'] ?? 'Not Assigned'); ?></td>
                    </tr>
                    <tr>
                        <th>Academic Year</th>
                        <td><?php echo htmlspecialchars($student['current_year']); ?></td>
                    </tr>
                </table>
            </div>
        </div>

        <h5>Comprehensive Fee Statement by Level</h5>
        <?php
        $current_level = null;
        $level_totals = [];
        
        // Store all fees in an array first and group by level
        $fees_by_level = [];
        while ($fee = $fees->fetch_assoc()) {
            $fees_by_level[$fee['level']][] = $fee;
        }
        
        // Sort levels naturally
        ksort($fees_by_level);
        
        foreach ($fees_by_level as $level => $level_fees):
            $level_total = [
                'fees' => 0,
                'paid' => 0,
                'balance' => 0
            ];
        ?>
        <div class="level-section mb-4">
            <h6 class="bg-light p-2"><?php echo htmlspecialchars($level); ?></h6>
            <div class="table-responsive">
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th>Academic Year</th>
                            <th>Fee Type</th>
                            <th>Amount (GHs)</th>
                            <th>Paid (GHs)</th>
                            <th>Balance (GHs)</th>
                            <th>Status</th>
                            <th>Payment History</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($level_fees as $fee): 
                            $balance = $fee['amount'] - $fee['paid_amount'];
                            $level_total['fees'] += $fee['amount'];
                            $level_total['paid'] += $fee['paid_amount'];
                            $level_total['balance'] += $balance;
                            
                            // Parse payment history
                            $payment_history = [];
                            if (!empty($fee['payment_history'])) {
                                $payments = explode(';;', $fee['payment_history']);
                                foreach ($payments as $payment) {
                                    list($date, $amount, $method, $ref, $status) = explode('|', $payment);
                                    $payment_history[] = [
                                        'date' => $date,
                                        'amount' => $amount,
                                        'method' => $method,
                                        'reference' => $ref,
                                        'status' => $status
                                    ];
                                }
                            }
                            
                            // Calculate totals for this section
                            if ($fee['academic_year_id'] == $current_academic_year['id']) {
                                // Current year fees
                                $financial_summary['provisional_fees'] += $fee['amount'];
                            } else if ($fee['academic_year_id'] < $current_academic_year['id']) {
                                // Previous years' balance
                                $financial_summary['existing_balance'] += ($fee['amount'] - $fee['paid_amount']);
                            }
                        ?>
                            <tr>
                                <td><?php echo htmlspecialchars($fee['year_name']); ?></td>
                                <td><?php echo htmlspecialchars($fee['fee_type']); ?></td>
                                <td class="text-end"><?php echo number_format($fee['amount'], 2); ?></td>
                                <td class="text-end"><?php echo number_format($fee['paid_amount'], 2); ?></td>
                                <td class="text-end"><?php echo number_format($balance, 2); ?></td>
                                <td>
                                    <span class="badge bg-<?php 
                                        echo $fee['status'] === 'paid' ? 'success' : 
                                            ($fee['status'] === 'partial' ? 'warning' : 'danger'); 
                                    ?>">
                                        <?php echo ucfirst($fee['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if (!empty($payment_history)): ?>
                                        <div class="payment-history">
                                            <?php foreach ($payment_history as $payment): ?>
                                                <div class="payment-item <?php echo $payment['status']; ?>">
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <span class="payment-date">
                                                            <i class="fas fa-calendar-alt me-1"></i>
                                                            <?php echo date('d M Y h:i A', strtotime($payment['date'])); ?>
                                                        </span>
                                                        <span class="payment-badge <?php echo $payment['status']; ?>">
                                                            <?php echo ucfirst($payment['status']); ?>
                                                        </span>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center mt-1">
                                                        <small class="text-muted">
                                                            <i class="fas fa-money-bill-wave me-1"></i>
                                                            GHs <?php echo number_format($payment['amount'], 2); ?>
                                                            via <?php echo $payment['method']; ?>
                                                        </small>
                                                        <small class="text-muted">
                                                            <i class="fas fa-hashtag me-1"></i>
                                                            <?php echo $payment['reference']; ?>
                                                        </small>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php else: ?>
                                        <small class="text-muted">No payments recorded</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="table-light fw-bold">
                            <td colspan="2"><?php echo htmlspecialchars($level); ?> Total</td>
                            <td class="text-end"><?php echo number_format($level_total['fees'], 2); ?></td>
                            <td class="text-end"><?php echo number_format($level_total['paid'], 2); ?></td>
                            <td class="text-end"><?php echo number_format($level_total['balance'], 2); ?></td>
                            <td colspan="2"></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endforeach; ?>

        <!-- Financial Summary -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Financial Summary</h5>
                <small class="text-muted">Currency: GH₵</small>
            </div>
            <div class="card-body">
                <table class="table">
                    <tbody>
                        <tr>
                            <th>Total Fees</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['total_fees'], 2); ?></td>
                        </tr>
                        <tr>
                            <th>Total Payments</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['total_payments'], 2); ?></td>
                        </tr>
                        <tr class="fw-bold <?php echo $financial_summary['overall_balance'] > 0 ? 'table-danger' : 'table-success'; ?>">
                            <th>Overall Balance</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['overall_balance'], 2); ?></td>
                        </tr>
                        <tr>
                            <th>Existing Balance (Previous Years)</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['existing_balance'], 2); ?></td>
                        </tr>
                        <tr>
                            <th>Current Year Fees (<?php echo htmlspecialchars($current_academic_year['year_name']); ?>)</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['provisional_fees'], 2); ?></td>
                        </tr>
                        <tr>
                            <th>Current Year Payments</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['current_year_payments'], 2); ?></td>
                        </tr>
                        <?php if (isset($financial_summary['current_year_payments_refunded']) && $financial_summary['current_year_payments_refunded'] > 0): ?>
                        <tr class="table-warning">
                            <th class="ps-4">Current Year Refunds</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['current_year_payments_refunded'], 2); ?></td>
                        </tr>
                        <?php endif; ?>
                        <tr class="table-active fw-bold">
                            <th>Balance Carry Forward</th>
                            <td class="text-end">GH₵ <?php echo number_format($financial_summary['balance_forward'], 2); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="footer">
            <div class="row">
                <div class="col-md-8">
                    <p class="mb-0"><strong>Note:</strong> This is an official fee statement from the KCOE Fee Payment System.</p>
                    <p class="text-muted">For any queries, please contact the accounts office.</p>
                </div>
                <div class="col-md-4 text-end">
                    <p class="mb-0">Document No: <?php echo date('Ymd') . '-' . $student['student_number']; ?></p>
                    <p class="text-muted">Generated by system</p>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>