<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify student access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student') {
    header('Location: ../login.php');
    exit();
}

// Get student details
$stmt = $conn->prepare("
    SELECT 
        s.*,
        u.email,
        p.program_name,
        ay.year_name as current_year,
        (
            SELECT COALESCE(SUM(ft.amount), 0)
            FROM student_fees sf
            JOIN fee_types ft ON sf.fee_type_id = ft.id
            WHERE sf.student_id = s.id
        ) as total_fees,
        (
            SELECT COALESCE(SUM(p.amount), 0)
            FROM payments p
            JOIN student_fees sf ON p.student_fee_id = sf.id
            WHERE sf.student_id = s.id
            AND p.status = 'confirmed'
        ) as total_paid
    FROM students s
    JOIN users u ON s.user_id = u.id
    LEFT JOIN programs p ON s.program_id = p.id
    LEFT JOIN academic_years ay ON ay.is_current = 1
    WHERE s.user_id = ?
");

$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$student = $stmt->get_result()->fetch_assoc();

// Get all payments for the current student
$stmt = $conn->prepare("
    SELECT 
        p.*,
        ft.name as fee_type,
        ay.year_name as academic_year,
        ft.amount as fee_amount,
        s.student_number,
        s.id as student_id,
        (
            SELECT COALESCE(SUM(p2.amount), 0)
            FROM payments p2
            JOIN student_fees sf2 ON p2.student_fee_id = sf2.id
            WHERE sf2.student_id = sf.student_id
            AND p2.status = 'confirmed'
            AND p2.created_at <= p.created_at
            AND sf2.fee_type_id = ft.id
        ) as paid_amount,
        ft.amount - (
            SELECT COALESCE(SUM(p2.amount), 0)
            FROM payments p2
            JOIN student_fees sf2 ON p2.student_fee_id = sf2.id
            WHERE sf2.student_id = sf.student_id
            AND p2.status = 'confirmed'
            AND p2.created_at <= p.created_at
            AND sf2.fee_type_id = ft.id
        ) as running_balance,
        ft.amount as total_fee_amount,
        (
            SELECT COALESCE(SUM(amount), 0)
            FROM payments
            WHERE student_fee_id = sf.id
            AND status = 'confirmed'
            AND payment_date <= p.payment_date
        ) as cumulative_paid,
        (
            SELECT COALESCE(SUM(p2.amount), 0)
            FROM payments p2
            WHERE p2.student_fee_id = sf.id
            AND p2.status = 'confirmed'
        ) as total_paid_for_fee
    FROM payments p
    JOIN student_fees sf ON p.student_fee_id = sf.id
    JOIN students s ON sf.student_id = s.id
    JOIN users u ON s.user_id = u.id
    JOIN fee_types ft ON sf.fee_type_id = ft.id
    JOIN academic_years ay ON sf.academic_year_id = ay.id
    WHERE sf.student_id = ? 
    AND u.id = ?
    ORDER BY p.payment_date DESC, p.id DESC
");

$stmt->bind_param("ii", $student['id'], $_SESSION['user_id']);
$stmt->execute();
$payments = $stmt->get_result();

// Set page title and include header
$page_title = "Payment History";
require_once '../includes/header.php';
?>

<div class="container-fluid py-4">
    <!-- Student Information -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="card-title mb-0">Student Information</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Student Number:</strong> <?php echo htmlspecialchars($student['student_number']); ?></p>
                            <p><strong>Name:</strong> <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></p>
                            <p><strong>Program:</strong> <?php echo htmlspecialchars($student['program_name'] ?? 'Not Assigned'); ?></p>
                            <p><strong>Year Level:</strong> <?php echo htmlspecialchars($student['year_level'] ?? 'Not Set'); ?></p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Total Fees:</strong> <?php echo formatMoney($student['total_fees']); ?></p>
                            <p><strong>Total Paid:</strong> <?php echo formatMoney($student['total_paid']); ?></p>
                            <p><strong>Outstanding Balance:</strong> <?php echo formatMoney($student['total_fees'] - $student['total_paid']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment History -->
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="card-title mb-0">Payment History</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Fee Type</th>
                                    <th>Academic Year</th>
                                    <th>Amount Paid</th>
                                    <th>Running Balance</th>
                                    <th>Payment Method</th>
                                    <th>Reference</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($payment = $payments->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo date('M d, Y', strtotime($payment['payment_date'] ?: $payment['created_at'])); ?></td>
                                    <td><?php echo htmlspecialchars($payment['fee_type']); ?></td>
                                    <td><?php echo htmlspecialchars($payment['academic_year']); ?></td>
                                    <td><?php echo formatMoney($payment['amount']); ?></td>
                                    <td>
                                        <?php 
                                        $runningBalance = $payment['running_balance'];
                                        if ($runningBalance > 0) {
                                            echo '<span class="text-danger">' . formatMoney($runningBalance) . '</span>';
                                        } else {
                                            echo '<span class="text-success">Paid</span>';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        $method = $payment['payment_method'];
                                        if ($method) {
                                            echo ucfirst(str_replace('_', ' ', htmlspecialchars($method)));
                                        } else {
                                            echo '<span class="text-muted">-</span>';
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($payment['reference_number']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            echo $payment['status'] === 'confirmed' ? 'success' : 
                                                ($payment['status'] === 'pending' ? 'warning' : 'danger'); 
                                        ?>">
                                            <?php echo ucfirst($payment['status']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Make New Payment Button -->
    <div class="row mt-4">
        <div class="col-md-12 text-end">
            <a href="submit_payment.php" class="btn btn-primary">
                <i class="fas fa-plus"></i> Make New Payment
            </a>
        </div>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>