<?php
require_once '../config/config.php';
require_once '../controllers/AuthController.php';
require_once '../includes/functions.php';

session_start();

// Verify student access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student') {
    header('Location: ../login.php');
    exit();
}

// Get student details
$stmt = $conn->prepare("SELECT s.id, s.student_number, s.first_name, s.last_name, p.program_name, s.year_level, u.email 
    FROM students s 
    LEFT JOIN programs p ON s.program_id = p.id 
    LEFT JOIN users u ON s.user_id = u.id 
    WHERE s.user_id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$student = $stmt->get_result()->fetch_assoc();

// Get unpaid fees with improved query that shows fees with remaining balance
$query = "SELECT 
    sf.id, 
    ft.amount as original_amount, 
    sf.due_date, 
    sf.status, 
    ft.name as fee_type, 
    ay.year_name,
    COALESCE((
        SELECT SUM(amount) 
        FROM payments 
        WHERE student_fee_id = sf.id 
        AND status = 'confirmed'
    ), 0) as paid_amount
FROM student_fees sf 
JOIN fee_types ft ON sf.fee_type_id = ft.id 
JOIN academic_years ay ON sf.academic_year_id = ay.id 
WHERE sf.student_id = ? 
AND (
    COALESCE((
        SELECT SUM(amount) 
        FROM payments 
        WHERE student_fee_id = sf.id 
        AND status = 'confirmed'
    ), 0) < ft.amount
)
ORDER BY ay.year_name DESC, ft.name";

$stmt = $conn->prepare($query);
$stmt->bind_param("i", $student['id']);
$stmt->execute();
$unpaid_fees = $stmt->get_result();

// Initialize messages
$success_message = '';
$error_message = '';

// Handle payment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $fee_id = $_POST['fee_id'] ?? '';
        $amount = floatval($_POST['amount'] ?? 0);
        $payment_method = $_POST['payment_method'] ?? '';
        
        if (empty($fee_id) || empty($amount) || empty($payment_method)) {
            throw new Exception("All fields are required.");
        }
        
        // Initialize Paystack payment
        if ($payment_method === 'paystack') {
            $reference = 'KCOE_' . time() . '_' . $student['id'] . '_' . $fee_id;
            
            // Insert initial payment record
            $stmt = $conn->prepare("INSERT INTO payments (student_fee_id, amount, payment_method, reference_number, status) VALUES (?, ?, ?, ?, 'pending')");
            $stmt->bind_param("idss", $fee_id, $amount, $payment_method, $reference);
            $stmt->execute();
            
            // Redirect to initialize_paystack.php
            header('Location: ../payment/initialize_paystack.php?reference=' . $reference);
            exit();
        }
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

// Set page title and include header
$page_title = "Submit Payment";
require_once '../includes/header.php';
?>

<div class="container py-4">
    <div class="row justify-content-center">
        <div class="col-md-8">
            <div class="card shadow">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Submit New Payment</h5>
                </div>
                <div class="card-body">
                    <?php if ($success_message): ?>
                        <div class="alert alert-success"><?php echo $success_message; ?></div>
                    <?php endif; ?>

                    <?php if ($error_message): ?>
                        <div class="alert alert-danger"><?php echo $error_message; ?></div>
                    <?php endif; ?>

                    <form method="POST" action="" id="paymentForm" class="needs-validation" novalidate>
                        <div class="mb-3">
                            <label class="form-label">Select Fee to Pay</label>
                            <?php if ($unpaid_fees->num_rows == 0): ?>
                                <div class="alert alert-info">
                                    No unpaid fees found. Please contact the administrator if this seems incorrect.
                                </div>
                            <?php endif; ?>
                            
                            <select name="fee_id" class="form-select" required>
                                <option value="">Select a fee...</option>
                                <?php while ($fee = $unpaid_fees->fetch_assoc()): 
                                    $remaining = $fee['original_amount'] - $fee['paid_amount'];
                                    if ($remaining <= 0) continue;
                                ?>
                                    <option value="<?php echo $fee['id']; ?>" 
                                            data-remaining="<?php echo $remaining; ?>">
                                        <?php echo htmlspecialchars($fee['fee_type'] . ' - ' . $fee['year_name']); ?>
                                        (Remaining: <?php echo formatMoney($remaining); ?>)
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <div class="invalid-feedback">Please select a fee to pay.</div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount</label>
                            <input type="number" name="amount" class="form-control" step="0.01" required>
                            <div class="invalid-feedback">Please enter a valid amount.</div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Payment Method</label>
                            <select name="payment_method" class="form-select" required>
                                <option value="">Select payment method...</option>
                                <option value="paystack">Paystack</option>
                            </select>
                            <div class="invalid-feedback">Please select a payment method.</div>
                        </div>

                        <button type="submit" class="btn btn-primary">Submit Payment</button>
                        <a href="dashboard.php" class="btn btn-secondary">Back to Dashboard</a>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Form validation
(function () {
    'use strict'
    var form = document.getElementById('paymentForm')
    var feeSelect = form.querySelector('[name="fee_id"]')
    var amountInput = form.querySelector('[name="amount"]')

    feeSelect.addEventListener('change', function() {
        if (this.selectedIndex > 0) {
            var option = this.options[this.selectedIndex]
            var remaining = parseFloat(option.dataset.remaining)
            amountInput.max = remaining
            amountInput.placeholder = `Enter amount (max: GHs ${remaining.toFixed(2)})`
        }
    })

    form.addEventListener('submit', function (event) {
        if (!form.checkValidity()) {
            event.preventDefault()
            event.stopPropagation()
        }
        
        if (feeSelect.selectedIndex > 0) {
            var option = feeSelect.options[feeSelect.selectedIndex]
            var remaining = parseFloat(option.dataset.remaining)
            var amount = parseFloat(amountInput.value)
            
            if (amount > remaining) {
                event.preventDefault()
                alert('Amount cannot exceed the remaining balance of GHs ' + remaining.toFixed(2))
                return false
            }
        }
        
        form.classList.add('was-validated')
    }, false)
})()
</script>

<?php require_once '../includes/footer.php'; ?>